package ru.yandex.autotests.directapi.dynamictextadtargets.setbids;

import java.util.Arrays;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsFeatures;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsLogins;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.SetBidsItemMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.SetBidsRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageAddItemMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.general.SetBidsExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;

/**
 * Created by semkagtn on 24.11.15.
 * https://st.yandex-team.ru/TESTIRT-7700
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-43826")
@Features(DynamicTextAdTargetsFeatures.SET_BIDS)
@Description("Проверка минимальных и максимальных значений для Bid при разных валютах")
@RunWith(Parameterized.class)
public class SetBidsBidTresholdsTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public String login;

    @Parameterized.Parameter(1)
    public Currency currency;

    @Parameterized.Parameter(2)
    public String stringCurrency;

    @Parameterized.Parameters(name = "{0}")
    public static Iterable<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {Logins.LOGIN_RUB, Currency.RUB,
                        MoneyCurrency.get(Currency.RUB).getNamePluralI(api.locale())},
                {Logins.LOGIN_UAH, Currency.UAH,
                        MoneyCurrency.get(Currency.UAH).getNameCasusDativus(api.locale())},
                {Logins.LOGIN_KZT, Currency.KZT,
                        MoneyCurrency.get(Currency.KZT).getNameCasusDativus(api.locale())},
                {Logins.LOGIN_EUR, Currency.EUR,
                        MoneyCurrency.get(Currency.EUR).getAbbreviation(api.locale())},
                {Logins.LOGIN_USD, Currency.USD,
                        MoneyCurrency.get(Currency.USD).getNameCasusDativus(api.locale())},
                {Logins.LOGIN_CHF, Currency.CHF,
                        MoneyCurrency.get(Currency.CHF).getNameCasusDativus(api.locale())},
                {Logins.LOGIN_TRY, Currency.TRY,
                        MoneyCurrency.get(Currency.TRY).getNamePluralI(api.locale())},
                {DynamicTextAdTargetsLogins.CLIENT_BYN, Currency.BYN,
                        MoneyCurrency.get(Currency.BYN).getNamePluralI(api.locale())}
        });
    }

    private Long webpageId;

    @Before
    @Step("Подготовка данных для теста")
    public void prepareData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(login);

        api.as(login);
        Long campaignId = api.userSteps.campaignSteps().addDefaultDynamicTextCampaign();
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(campaignId);
        webpageId = api.userSteps.dynamicTextAdTargetsSteps().addWebpage(new WebpageAddItemMap()
                .defaultWebpage(adGroupId)
                .withBid(5_000_000L));
    }

    @Test
    public void minimumValue() {
        long minValue = MoneyCurrency.get(currency).getLongMinPrice().longValue();
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap()
                        .withBids(new SetBidsItemMap()
                                .withId(webpageId)
                                .withBid(minValue)),
                SetBidsExpectedResult.successWithId());
    }

    @Test
    public void maximumValue() {
        long maxValue = MoneyCurrency.get(currency).getLongMaxPrice().longValue();
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap()
                        .withBids(new SetBidsItemMap()
                                .withId(webpageId)
                                .withBid(maxValue)),
                SetBidsExpectedResult.successWithId());
    }

    @Test
    public void lessThanMinimumValue() {
        Money minValue = MoneyCurrency.get(currency).getMinPrice();
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap()
                        .withBids(new SetBidsItemMap()
                                .withId(webpageId)
                                .withBid(minValue.getPrevious().bidLong().longValue())),
                SetBidsExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetails.BID_VALUE_IN_FIELD_MUST_NOT_BE_LESS,
                                capitalize(SetBidsItemMap.BID), stringCurrency, minValue.bidLong().longValue())));
    }

    @Test
    public void greaterThanMaximumValue() {
        Money maxValue = MoneyCurrency.get(currency).getMaxPrice();
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap()
                        .withBids(new SetBidsItemMap()
                                .withId(webpageId)
                                .withBid(maxValue.getNext().bidLong().longValue())),
                SetBidsExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetails.BID_VALUE_IN_FIELD_MUST_NOT_BE_MORE,
                                capitalize(SetBidsItemMap.BID), stringCurrency, maxValue.bidLong().longValue())));
    }
}
