package ru.yandex.autotests.directapi.dynamictextadtargets.suspend;

import com.yandex.direct.api.v5.dynamictextadtargets.StringConditionOperatorEnum;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageConditionOperandEnum;
import org.apache.commons.lang3.StringUtils;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsFeatures;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsLogins;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.SuspendRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageAddItemMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageConditionMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageGetItemMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by ginger on 13.11.15.
 * https://st.yandex-team.ru/TESTIRT-6666
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-43826")
@Features(DynamicTextAdTargetsFeatures.SUSPEND)
@Description("Проверка списания баллов при остановки условия нацеливания")
public class SuspendUnitsTest {

    private static final String LOGIN = DynamicTextAdTargetsLogins.UNITS_SUSPEND_CLIENT;
    private static Long cid;
    private Long pid;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int CALL_COST = 10;
    private static final int WEBPAGE_COST = 1;
    private static final int ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR_COST = 50;

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        api.as(LOGIN);
        cid = api.userSteps.campaignSteps().addDefaultDynamicTextCampaign(LOGIN);
        pid = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(cid, LOGIN);
    }

    @Test
    @Description("Отключение одного условия")
    public void suspendOneWebpage() {
        long webpageID = api.userSteps.dynamicTextAdTargetsSteps().addWebpage(
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withBid(30000000l)
        );
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsSuspend(
                new SuspendRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(webpageID))
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + WEBPAGE_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Отключение двух условий")
    public void suspendTwoWebpages() {
        long fistWebpageID = api.userSteps.dynamicTextAdTargetsSteps().addWebpage(
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withConditions(
                                new WebpageConditionMap()
                                        .withArguments("картоха")
                                        .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                        .withOperand(WebpageConditionOperandEnum.PAGE_CONTENT)
                        )
                        .withBid(30000000l)
        );
        long secondWebpageID = api.userSteps.dynamicTextAdTargetsSteps().addWebpage(
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name2")
                        .withConditions(
                                new WebpageConditionMap()
                                        .withArguments("картоха")
                                        .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                        .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                        )
                        .withBid(30000000l)
        );
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsSuspend(
                new SuspendRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(fistWebpageID, secondWebpageID))
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + WEBPAGE_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Отключение одного невалидного условия")
    public void suspendOneInvalidWebpage() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSuspend(
                new SuspendRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(1l)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.WEBPAGE_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Отключение двух невалидных условий")
    public void suspendTwoInvalidWebpages() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSuspend(
                new SuspendRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(0l, 1l)),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FIELD_MUST_BE_POSITIVE_INTEGER,
                        StringUtils.capitalize(WebpageGetItemMap.ID))),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.WEBPAGE_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Отключение одного валидного и одного невалидного условия")
    public void suspendOneValidAndOneInvalidWebpages() {
        long webpageID = api.as(LOGIN).userSteps.dynamicTextAdTargetsSteps().addWebpage(
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withBid(30000000l)
        );
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSuspend(
                new SuspendRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(webpageID, 1l)),
                ExpectedResult.success(webpageID),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.WEBPAGE_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST + WEBPAGE_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Невалидный запрос на отключение условий")
    public void invalidRequest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsSuspend(
                new SuspendRequestMap()
                        .withSelectionCriteria(null),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        StringUtils.capitalize(SuspendRequestMap.SELECTION_CRITERIA)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = COMMON_REQUEST_ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Отключение условия, когда у клиента нету баллов")
    public void suspendWebpageWhenClientHasZeroUnits() {
        long webpageID = api.userSteps.dynamicTextAdTargetsSteps().addWebpage(
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withBid(30000000l)
        );
        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("у клиента нету баллов", units, equalTo(0));

        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsSuspend(
                new SuspendRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(webpageID)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
    }
}
