package ru.yandex.autotests.directapi.dynamictextadtargets.suspend;

import com.yandex.direct.api.v5.dynamictextadtargets.AddResponse;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.CampaignFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsFeatures;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsLogins;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.SuspendRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageAddItemMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by ginger on 13.11.15.
 * https://st.yandex-team.ru/TESTIRT-6666
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-43826")
@Features(DynamicTextAdTargetsFeatures.SUSPEND)
@Description("Проверка у группы времени последнего изменения и статуса синхронизации с БК при остановки условия нацеливания")
public class SuspendWebpagesAdGroupLastChangeAndBsSyncedTest {
    private static LogSteps log = LogSteps.getLogger(SuspendWebpagesAdGroupLastChangeAndBsSyncedTest.class);
    private static final String LOGIN = DynamicTextAdTargetsLogins.SINGLE_CLIENT_SUSPEND;

    private static DateTime lastChange = DateTime.now().minusDays(5);
    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");
    private static Long pid;
    private static AddResponse response;
    private static Long cid;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @Rule
    public Trashman trashman = new Trashman(api);

    @BeforeClass
    public static void prepareGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        cid = api.userSteps.campaignSteps().addDefaultDynamicTextCampaign(LOGIN);
        pid = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(cid);
        response =  api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap()
                        .withWebpages(new WebpageAddItemMap()
                                .withAdGroupId(pid)
                                .withName("Name")
                                .withBid(30000000l)
                        ),
                ExpectedResult.success()
        );

        log.info("Промодерируем фейково");
        api.userSteps.campaignFakeSteps().makeCampaignModerated(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);

        log.info("Выставим статусы синхронизации с БК фейково");
        api.userSteps.campaignFakeSteps().setBSSynced(cid, true);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(pid, Status.YES);

        log.info("Сдвинем время последнего изменения на несколько дней");
        api.userSteps.campaignFakeSteps().setLastChange(cid, lastChange.toString());
        api.userSteps.groupFakeSteps().setLastChange(pid, lastChange.toString());

        log.info("Возобновим условие нацеливания");
        api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsSuspend(
                new SuspendRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(
                        response.getAddResults().get(0).getId()
                ))
        );
    }

    @Test
    public void campaignStatusBsSyncedTest() {
        CampaignFakeInfo campaignInfo = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(cid);

        assertThat("статус синхронизации с БК у кампании после остановки условия нацеливания",
                campaignInfo.getStatusBsSynced(), equalTo(Status.YES));
    }

    @Test
    public void groupStatusBsSyncedTest() {
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(pid);

        assertThat("статус синхронизации с БК у группы после остановки условия нацеливания",
                groupInfo.getStatusBsSynced(), equalTo(Status.NO));
    }

    @Test
    public void campaignLastChangeTest() {
        CampaignFakeInfo campaignInfo = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(cid);
        assumeThat("время последнего изменения кампании", campaignInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(campaignInfo.getLastChange());

        assertThat("время последнего изменения кампании после остановки условия нацеливания",
                date, equalTo(lastChange.toLocalDate()));
    }

    @Test
    public void groupLastChangeTest() {
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(pid);
        assumeThat("время последнего изменения группы", groupInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(groupInfo.getLastChange());

        assertThat("время последнего изменения группы после остановки условия нацеливания",
                date, equalTo(LocalDate.now()));
    }
}
