package ru.yandex.autotests.directapi.keywords.add;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsBaseRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by gerdler on 28.03.17.
 * https://st.yandex-team.ru/DIRECT-64666
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Description("Проверка ставок по умолчанию после создания автотаргетинга")
@Issue("https://st.yandex-team.ru/DIRECT-64386")
@RunWith(Parameterized.class)
public class AddAutotargetingAndCheckPriceTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public String login;

    @Parameterized.Parameter(1)
    public Currency currency;

    @Parameterized.Parameters(name = "Login = {0}, Currency = {1}")
    public static Collection data() {
        Object[][] data = new Object[][]{
                {Logins.LOGIN_FOR_RUB, Currency.RUB},
                {Logins.LOGIN_FOR_USD, Currency.USD},
                {Logins.LOGIN_FOR_UAH, Currency.UAH},
                {Logins.LOGIN_FOR_EUR, Currency.EUR},
                {Logins.LOGIN_FOR_KZT, Currency.KZT},
                {Logins.LOGIN_FOR_CHF, Currency.CHF},
                {Logins.LOGIN_FOR_TRY, Currency.TRY},
                {Logins.CLIENT_BYN, Currency.BYN}
        };
        return Arrays.asList(data);
    }

    private BidsBaseRecord bidsBaseRecord;
    private BigDecimal minPrice;
    private BigDecimal defaultPrice;

    @Before
    public void prepareData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(login);
        Long campaignId = api.as(login).userSteps.campaignSteps().addDefaultTextCampaign(login);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        Long autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);
        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(login);
        bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(autotargetingId);
        assumeThat("из таблицы bids_base получена информация о бесфразном таргетинге", bidsBaseRecord, notNullValue());
        minPrice = MoneyCurrency.get(currency).getMinPrice().bigDecimalValue().setScale(2);
        defaultPrice = MoneyCurrency.get(currency).getDefaultPrice().bigDecimalValue().setScale(2);
    }

    @Test
    public void checkPrices() {
        assertThat("бесфразный таргетинг создался с минимальной ставкой на поиске", bidsBaseRecord.getPrice(),
                equalTo(minPrice));
        assertThat("бесфразный таргетинг создался с минимальной ставкой в сети", bidsBaseRecord.getPriceContext(),
                equalTo(defaultPrice));
    }
}
