package ru.yandex.autotests.directapi.keywords.add;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.general.PriorityEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsBaseRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.bslogs.AutoBudgetPriority;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by gerdler on 29.03.17.
 * https://st.yandex-team.ru/DIRECT-64666
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Description("Проверка значений ставок по умолчанию при создании автотаргетинга")
@Issue("https://st.yandex-team.ru/DIRECT-64386")
@RunWith(Parameterized.class)
public class AddAutotargetingBidContextBidAndStrategyPriorityDefaultValuesTest {

    private static final Integer DEFAULT_AUTOBUDGET_PRIORITY =
            Integer.valueOf(AutoBudgetPriority.MEDIUM.transportValue());
    private static final BigDecimal ZERO_BID = BigDecimal.ZERO.setScale(2);

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public Currency currency;

    @Parameterized.Parameter(1)
    public String login;

    @Parameterized.Parameters(name = "Currency = {0}")
    public static Iterable<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {Currency.RUB, Logins.LOGIN_FOR_RUB},
                {Currency.UAH, Logins.LOGIN_FOR_UAH},
                {Currency.KZT, Logins.LOGIN_FOR_KZT},
                {Currency.EUR, Logins.LOGIN_FOR_EUR},
                {Currency.USD, Logins.LOGIN_FOR_USD},
                {Currency.CHF, Logins.LOGIN_FOR_CHF},
                {Currency.TRY, Logins.LOGIN_FOR_TRY},
                {Currency.BYN, Logins.CLIENT_BYN}
        });
    }

    private Long adGroupId;
    private Long maxBid;
    private BigDecimal minBid;
    private BigDecimal defaultBid;
    private BigDecimal maxBidBigDecimal;

    private DirectJooqDbSteps jooqDbSteps;

    @Before
    @Step("Подготовка данных для теста")
    public void createAdGroup() {
        api.as(login).userSteps.clientFakeSteps().fakeClearClientSpentUnits(login);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(login);
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        MoneyCurrency moneyCurrency = MoneyCurrency.get(currency);
        minBid = moneyCurrency.getMinPrice().bigDecimalValue().setScale(2);
        defaultBid = MoneyCurrency.get(currency).getDefaultPrice().bigDecimalValue().setScale(2);
        maxBid = moneyCurrency.getLongMaxPrice().longValue();
        maxBidBigDecimal = moneyCurrency.getMaxPrice().bigDecimalValue().setScale(2);
    }

    @Test
    public void createAutotargetingDefaultBid() {
        Long autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);

        BidsBaseRecord bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(autotargetingId);
        assertThat("ставки по умолчанию установились верно",
                bidsBaseRecord.intoMap(),
                beanDifferV5(
                        new BidsBaseRecord().setBidId(autotargetingId)
                                .setPrice(minBid)
                                .setPriceContext(defaultBid)
                                .setAutobudgetpriority(DEFAULT_AUTOBUDGET_PRIORITY)
                                .intoMap()
                ).useCompareStrategy(DefaultCompareStrategies.onlyExpectedFields())
        );
    }

    @Test
    public void createAutotargetingDefaultBidInAdGroupWithKeywords() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw1").withBid(maxBid).withContextBid(maxBid),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw2").withBid(maxBid).withContextBid(maxBid)
        );
        assumeThat("добавлены две фразы", keywordIDs, hasSize(2));
        Long autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);

        BidsBaseRecord bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(autotargetingId);
        assertThat("ставки по умолчанию установились верно",
                bidsBaseRecord.intoMap(),
                beanDifferV5(
                        new BidsBaseRecord().setBidId(autotargetingId)
                                .setPrice(maxBidBigDecimal)
                                .setPriceContext(maxBidBigDecimal)
                                .setAutobudgetpriority(DEFAULT_AUTOBUDGET_PRIORITY)
                                .intoMap()
                ).useCompareStrategy(DefaultCompareStrategies.onlyExpectedFields())
        );
    }

    @Test
    public void rесreateAutotargetingDefaultBid() {
        Long autotargetingId = api.userSteps.keywordsSteps()
                .addAutotargetingWithBidsAndStrategyPriority(adGroupId, maxBid, maxBid, PriorityEnum.HIGH);
        api.userSteps.keywordsSteps().keywordsDelete(autotargetingId);

        autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);

        BidsBaseRecord bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(autotargetingId);

        assertThat("ставки по умолчанию установились верно",
                bidsBaseRecord.intoMap(),
                beanDifferV5(
                        new BidsBaseRecord().setBidId(autotargetingId)
                                .setPrice(minBid)
                                .setPriceContext(defaultBid)
                                .setAutobudgetpriority(DEFAULT_AUTOBUDGET_PRIORITY)
                                .intoMap()
                ).useCompareStrategy(DefaultCompareStrategies.onlyExpectedFields())
        );
    }

    @Test
    public void rесreateAutotargetingDefaultBidInAdGroupWithKeywords() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw1").withBid(maxBid).withContextBid(maxBid),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw2").withBid(maxBid).withContextBid(maxBid)
        );
        assumeThat("добавлены две фразы", keywordIDs, hasSize(2));
        Long autotargetingId = api.userSteps.keywordsSteps()
                .addAutotargetingWithBidsAndStrategyPriority(adGroupId, maxBid, maxBid, PriorityEnum.HIGH);
        api.userSteps.keywordsSteps().keywordsDelete(autotargetingId);

        autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);

        BidsBaseRecord bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(autotargetingId);

        assertThat("ставки по умолчанию установились верно",
                bidsBaseRecord.intoMap(),
                beanDifferV5(
                        new BidsBaseRecord().setBidId(autotargetingId)
                                .setPrice(maxBidBigDecimal)
                                .setPriceContext(maxBidBigDecimal)
                                .setAutobudgetpriority(DEFAULT_AUTOBUDGET_PRIORITY)
                                .intoMap()
                ).useCompareStrategy(DefaultCompareStrategies.onlyExpectedFields())
        );
    }
}
