package ru.yandex.autotests.directapi.keywords.add;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.campaigns.TextCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.TextCampaignSearchStrategyTypeEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.model.campaigns.MetrikaGoals;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.directapi.model.api5.general.ExpectedResult.success;
import static ru.yandex.autotests.directapi.model.api5.general.ExpectedResult.warnings;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by gerdler on 29.03.17.
 * https://st.yandex-team.ru/DIRECT-64666
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Description("Проверка выставления ставок (Bid) на автотаргетинг при его создании для разных стратегий в кампании")
@Issue("https://st.yandex-team.ru/DIRECT-64386")
@RunWith(Parameterized.class)
public class AddAutotargetingBidDifferentStrategiesTest {

    private static final String CLIENT = KeywordsLogins.SINGLE_CLIENT_ADD;
    private static final Currency CURRENCY = Currency.RUB;
    private static final Long MAX_BID = MoneyCurrency.get(CURRENCY).getLongMaxPrice().longValue();
    private static final Long MIN_BID = MoneyCurrency.get(CURRENCY).getLongMinPrice().longValue();
    private static final Long BID = MoneyCurrency.get(CURRENCY).getLongMinPrice().longValue() * 10L;
    private static final Long GOAL_ID = MetrikaGoals.getGoalForLogin(CLIENT);
    private static final BigDecimal DEFAULT_AUTOTARGETING_BID = new BigDecimal(15000.2).setScale(2,BigDecimal.ROUND_DOWN);
    private static final BigDecimal ZERO_BID = BigDecimal.ZERO.setScale(2, BigDecimal.ROUND_DOWN);

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public TextCampaignSearchStrategyTypeEnum searchStrategyEnum;

    @Parameterized.Parameter(1)
    public TextCampaignNetworkStrategyTypeEnum networkStrategyEnum;

    @Parameterized.Parameter(2)
    public TextCampaignSearchStrategyMap searchStrategy;

    @Parameterized.Parameter(3)
    public TextCampaignNetworkStrategyMap networkStrategy;

    @Parameterized.Parameter(4)
    public ExpectedResult expectedResult;

    @Parameterized.Parameter(5)
    public BigDecimal expectedBid;

    @Parameterized.Parameters(name = "SearchStrategy = {0}, NetworkStrategy = {1}")
    public static Iterable<Object[]> data() {
        ExpectedResult autoStrategyWarning = warnings(new Notification(10160,
                Api5ErrorDetails.BID_WILL_NOT_APPLIED_BECAUSE_OF_AUTO_STRATEGY,
                capitalize(KeywordAddMap.BID)));

        Notification showDisabledNotification = new Notification(10160,
                Api5ErrorDetails.BID_WILL_NOT_APPLIED_BECAUSE_OF_SHOWS_DISABLED_ON_SEARCH,
                capitalize(KeywordAddMap.BID)
        );
        ExpectedResult showsDisabledWarning = warnings(showDisabledNotification);

        return Arrays.asList(new Object[][]{

                {TextCampaignSearchStrategyTypeEnum.AVERAGE_CPA,
                        TextCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT,
                        new TextCampaignSearchStrategyMap().defaultAverageCpa(CURRENCY, GOAL_ID),
                        new TextCampaignNetworkStrategyMap().defaultNetworkDefault(),
                        autoStrategyWarning,
                        ZERO_BID
                },
                {TextCampaignSearchStrategyTypeEnum.HIGHEST_POSITION,
                        TextCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT,
                        new TextCampaignSearchStrategyMap().defaultHighestPosition(),
                        new TextCampaignNetworkStrategyMap().defaultNetworkDefault(),
                        success(),
                        DEFAULT_AUTOTARGETING_BID
                },

                {TextCampaignSearchStrategyTypeEnum.HIGHEST_POSITION,
                        TextCampaignNetworkStrategyTypeEnum.MAXIMUM_COVERAGE,
                        new TextCampaignSearchStrategyMap().defaultHighestPosition(),
                        new TextCampaignNetworkStrategyMap().defaultMaximumCoverage(),
                        success(),
                        DEFAULT_AUTOTARGETING_BID
                },

                {TextCampaignSearchStrategyTypeEnum.SERVING_OFF,
                        TextCampaignNetworkStrategyTypeEnum.MAXIMUM_COVERAGE,
                        new TextCampaignSearchStrategyMap().defaultServingOff(),
                        new TextCampaignNetworkStrategyMap().defaultMaximumCoverage(),
                        showsDisabledWarning,
                        ZERO_BID
                },

                {TextCampaignSearchStrategyTypeEnum.SERVING_OFF,
                        TextCampaignNetworkStrategyTypeEnum.AVERAGE_CPA,
                        new TextCampaignSearchStrategyMap().defaultServingOff(),
                        new TextCampaignNetworkStrategyMap().defaultAverageCpc(CURRENCY),
                        showsDisabledWarning,
                        ZERO_BID
                },
        });
    }

    private Long adGroupId;

    private static DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT);

    @Before
    @Step("Подготовка данных для теста")
    public void prepareData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        jooqDbSteps.campaignsSteps().addCampMetrikaGoals(campaignId, GOAL_ID, 100L, 100L);
        api.userSteps.campaignSteps().campaignsUpdate(
                new UpdateRequestMap().withCampaigns(
                        new CampaignUpdateItemMap().withId(campaignId).withTextCampaign(
                                new TextCampaignUpdateItemMap().withBiddingStrategy(
                                        new TextCampaignStrategyMap()
                                                .withSearch(searchStrategy)
                                                .withNetwork(networkStrategy)
                                )
                        )
                )
        );
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Test
    public void createRelevanceMatchWithBid() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD, new AddRequestMap().withKeywords(
                        new KeywordAddMap()
                                .shortAutotargeting(adGroupId)
                                .withBid(BID)
                ),
                expectedResult
        );
    }

    @Test
    public void createRelevanceMatchWithDefaultBid() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw1").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw2").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw3").withBid(MIN_BID)
        );
        assumeThat("добавлены три фраза", keywordIDs, hasSize(3));
        Long autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);
        assertThat("установилась правильная ставка по умолчанию", jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(autotargetingId).getPrice(), equalTo(expectedBid));
    }
}
