package ru.yandex.autotests.directapi.keywords.add;

import java.math.BigDecimal;
import java.util.Arrays;

import com.yandex.direct.api.v5.keywords.AddResponse;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsBaseRecord;
import ru.yandex.autotests.direct.db.steps.BidsBaseSteps;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.features.FeatureNameEnum;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.keywords.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;

import static org.hamcrest.CoreMatchers.both;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Issue("https://st.yandex-team.ru/DIRECT-86894")
@Description("Проверка распространения автотаргетинга на сеть, для РМП кампаний")
@RunWith(Parameterized.class)
public class AddAutotargetingNewRmpFeatureExtendedRelevanceMatch {
    private static final String CLIENT_WITH_RM_FEATURE = KeywordsLogins.SINGLE_CLIENT_ADD_NEW_RMP_EXT_RM;
    private static final String CLIENT_WITHOUT_RM_FEATURE = KeywordsLogins.SINGLE_CLIENT_ADD_NEW_RMP;

    private static final Long MAX_BID = MoneyCurrency.get(Currency.RUB).getLongMaxPrice().longValue();
    private static final BigDecimal MAX_DB_BID = MoneyCurrency.get(Currency.RUB).getMaxPrice()
            .bigDecimalValue().setScale(2, BigDecimal.ROUND_UNNECESSARY);

    @ClassRule
    public static ApiSteps api = new ApiSteps();
    @Rule
    public Trashman trashman = new Trashman(api);

    @Parameterized.Parameter
    public String description;
    @Parameterized.Parameter(1)
    public String clientLogin;
    @Parameterized.Parameter(2)
    public boolean contextRelevanceMatchAllowed;

    private BidsBaseRecord bidsBaseByBidId;

    @Parameterized.Parameters(name = "{0}")
    public static Iterable<Object[]> stateValues() {
        return Arrays.asList(
                new TestCase("РМП для клиента с фиче-флагом 'context_relevance_match_allowed'")
                        .withClientLogin(CLIENT_WITH_RM_FEATURE)
                        .withContextRelevanceMatchAllowed(true)
                        .asArray(),
                new TestCase(
                        "РМП ля клиента с выключенным фиче-флагом 'context_relevance_match_allowed'")
                        .withClientLogin(CLIENT_WITHOUT_RM_FEATURE)
                        .withContextRelevanceMatchAllowed(false)
                        .asArray()
        );
    }

    @Before
    public void before() {
        api.as(clientLogin);
        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(clientLogin);
        Long clientId = Long.valueOf(User.get(clientLogin).getClientID());

        long contextRelevanceMatchAllowedFeatureId = jooqDbSteps.featuresSteps()
                .getFeatureId(FeatureNameEnum.CONTEXT_RELEVANCE_MATCH.getValue());

        if (contextRelevanceMatchAllowed) {
            jooqDbSteps.clientsSteps().setClientFeature(clientId, contextRelevanceMatchAllowedFeatureId, true);
        } else {
            assumeThat("Фиче-флаг context_relevance_match_allowed должен быть отключен",
                    jooqDbSteps.clientsSteps().isFeatureEnabled(clientId, contextRelevanceMatchAllowedFeatureId),
                    equalTo(false));
        }

        Long campaignId = api.as(clientLogin).userSteps.campaignSteps().addDefaultMobileAppCampaign();
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultMobileGroup(campaignId);

        AddResponse addResponse = api.userSteps.keywordsSteps().shouldGetResultOnAdd(
                clientLogin,
                new AddRequestMap().withKeywords(
                        new KeywordAddMap()
                                .autotargeting(adGroupId)
                                .withBid(MAX_BID)
                                .withContextBid(MAX_BID)
                ),
                ExpectedResult.success()
        );
        Long autotargetingId = addResponse.getAddResults().get(0).getId();

        bidsBaseByBidId = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(autotargetingId);
    }

    @Test
    public void relevanceMatchPriceContextMustBeSet() {
        assertThat("ставка на сеть сохранена в БД", bidsBaseByBidId.getPriceContext(), equalTo(MAX_DB_BID));
    }

    @Test
    public void relevanceMatchScopeMustContainNetworkAndSearch() {
        assertThat("в автотаргетинге не отключена 'сеть' и 'поиск'",
                bidsBaseByBidId.getOpts(),
                both(not(containsString(BidsBaseSteps.OPTS_NET_STOP)))
                        .and(not(containsString(BidsBaseSteps.OPTS_SEARCH_STOP))));
    }

    private static class TestCase {
        private String description;
        private String clientLogin;
        private boolean contextRelevanceMatchAllowed;

        private TestCase(String description) {
            this.description = description;
        }

        private TestCase withClientLogin(String clientLogin) {
            this.clientLogin = clientLogin;
            return this;
        }

        private TestCase withContextRelevanceMatchAllowed(boolean contextRelevanceMatchAllowed) {
            this.contextRelevanceMatchAllowed = contextRelevanceMatchAllowed;
            return this;
        }

        private Object[] asArray() {
            return new Object[]{description, clientLogin, contextRelevanceMatchAllowed};
        }
    }
}
