package ru.yandex.autotests.directapi.keywords.add;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.general.AutotargetingCategory;
import com.yandex.direct.api.v5.general.PriorityEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ClientsFeaturesRecord;
import ru.yandex.autotests.direct.utils.features.FeatureNameEnum;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.anyOf;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.directapi.model.api5.general.ExpectedResult.warnings;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by gerdler on 27.03.2017.
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Description("Позитивные сценарии добавления автотаргетинга")
public class AddAutotargetingPositiveTest {
    private static final String CLIENT = KeywordsLogins.SINGLE_CLIENT_ADD;
    private static final Long MAX_BID = MoneyCurrency.get(Currency.RUB).getLongMaxPrice().longValue();
    private static final Long MIN_BID = MoneyCurrency.get(Currency.RUB).getLongMinPrice().longValue();
    private static final Long DEFAULT_BID = MoneyCurrency.get(Currency.RUB).getLongDefaultPrice().longValue();
    private static final String OTHER_USER_PARAM = "other user param";
    private static final String ESCAPED_AUTOTARGETING_TEXT = "'---autotargeting";
    private static final Long DEFAULT_AUTOTARGETING_BID = 15000200000L;
    private static Long campaignId;
    private static Long adGroupId;
    private static Long bannerId;
    ExpectedResult javaExpectedResult;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void createCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT);

        // after DIRECT-83704
        // при включенном флаге context_relevance_match_allowed
        // не валидируем ContextPrice для автотаргетингов
        int shard = api.userSteps.clientFakeSteps().getUserShard(CLIENT);
        Long clientId = Long.valueOf(User.get(CLIENT).getClientID());
        long featureId = api.userSteps.getDirectJooqDbSteps().useShard(shard).featuresSteps()
                .getFeatureId(FeatureNameEnum.CONTEXT_RELEVANCE_MATCH.getValue());
        ClientsFeaturesRecord contextRelevanceMatchFeature =
                api.userSteps.getDirectJooqDbSteps().useShard(shard).clientsSteps()
                        .getClientsFeature(clientId, featureId);
    }

    @Before
    public void init() {
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        bannerId = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
    }

    @Test
    public void addAutotargetingInArchivedAdGroup() {
        api.userSteps.bannersFakeSteps().makeBannersModerated(bannerId);
        api.userSteps.adsSteps().adsArchive(bannerId);

        List<Long> keywordIds = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().shortAutotargeting(adGroupId)
        );

        assertThat("Должна быть возможность добавлять автотаргетинг в архивные группы",
                keywordIds, hasSize(1));
    }

    @Test
    public void addAutotargetingRequiredFieldsOnlyTest() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().shortAutotargeting(adGroupId)
        );

        assertThat("добавлен автотаргетинг с минимальным набором полей", keywordIDs, hasSize(1));
    }

    @Test
    public void addAutotargetingAllFieldsTest() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap()
                        .autotargeting(adGroupId)
                        .withBid(MAX_BID)
                        .withContextBid(MAX_BID)
                        .withStrategyPriority(PriorityEnum.NORMAL)
        );
        assertThat("добавлен автотаргетинг с максимальным набором полей", keywordIDs, hasSize(1));
    }

    @Test
    public void addExistentAutotargetingTest() {
        Long autotargetingId = api.userSteps.keywordsSteps()
                .addAutotargetingWithBidsAndStrategyPriority(adGroupId, MIN_BID, MIN_BID, PriorityEnum.LOW);
        List<KeywordGetItem> keywordGetItems = api.userSteps.keywordsSteps().keywordsGetById(autotargetingId);

        javaExpectedResult = warnings(
                new Notification(10161,
                        Api5ErrorDetails.FIELD_WILL_NOT_CHANGE_BECAUSE_OF_MANUAL_STRATEGY,
                        capitalize(KeywordAddMap.STRATEGY_PRIORITY)
                ),
                new Notification(7000, Api5ErrorDetails.ONLY_ONE_AUTOTARGETING_IN_ADGROUP)
        );
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(
                        new KeywordAddMap().autotargeting(adGroupId)
                                .withBid(MAX_BID)
                                .withContextBid(MAX_BID)
                                .withStrategyPriority(PriorityEnum.HIGH)
                                .withUserParam1(OTHER_USER_PARAM)
                                .withUserParam2(OTHER_USER_PARAM)
                ),
                javaExpectedResult);

        //DIRECT-152592: cортируем AutotargetingCategories
        List<KeywordGetItem> actualKeywordGetItems = api.userSteps.keywordsSteps().keywordsGetById(autotargetingId);
        assumeThat("", actualKeywordGetItems.size(), equalTo(1));
        actualKeywordGetItems.get(0).getAutotargetingCategories().getValue()
                .getItems().sort(Comparator.comparing(AutotargetingCategory::getCategory));
        assumeThat("", actualKeywordGetItems.size(), equalTo(1));
        keywordGetItems.get(0).getAutotargetingCategories().getValue()
                .getItems().sort(Comparator.comparing(AutotargetingCategory::getCategory));

        assertThat("параметры автотаргетинга не изменились",
                actualKeywordGetItems,
                beanDifferV5(keywordGetItems)
        );
    }

    @Test
    public void addExistentAutotargetingWithoutBidsTest() {
        Long autotargetingId = api.userSteps.keywordsSteps()
                .addAutotargetingWithBidsAndStrategyPriority(adGroupId, MIN_BID, MIN_BID, PriorityEnum.LOW);
        List<KeywordGetItem> keywordGetItems = api.userSteps.keywordsSteps().keywordsGetById(autotargetingId);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(
                        new KeywordAddMap().autotargeting(adGroupId)
                                .withUserParam1(OTHER_USER_PARAM)
                                .withUserParam2(OTHER_USER_PARAM)
                ),
                warnings(new Notification(7000, Api5ErrorDetails.ONLY_ONE_AUTOTARGETING_IN_ADGROUP)));

        //DIRECT-152592: cортируем AutotargetingCategories
        List<KeywordGetItem> actualKeywordGetItems = api.userSteps.keywordsSteps().keywordsGetById(autotargetingId);
        assumeThat("", actualKeywordGetItems.size(), equalTo(1));
        actualKeywordGetItems.get(0).getAutotargetingCategories().getValue()
                .getItems().sort(Comparator.comparing(AutotargetingCategory::getCategory));
        assumeThat("", actualKeywordGetItems.size(), equalTo(1));
        keywordGetItems.get(0).getAutotargetingCategories().getValue()
                .getItems().sort(Comparator.comparing(AutotargetingCategory::getCategory));

        assertThat("параметры автотаргетинга не изменились",
                actualKeywordGetItems,
                beanDifferV5(keywordGetItems)
        );
    }

    @Test
    public void recreateAutotargetingAndCheckIdTest() {
        Long autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);
        List<Long> deletedAutotargetingIds = api.userSteps.keywordsSteps().keywordsDelete(autotargetingId);
        assumeThat("удален автотаргетинг", deletedAutotargetingIds, hasSize(1));
        Long recreatedAutotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);
        assertThat("идентификатор бесфразного таргетинга при повторном добавлении не изменился",
                recreatedAutotargetingId, equalTo(autotargetingId));
    }

    @Test
    public void addAutotargetingWithEscapedTextTest() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword(ESCAPED_AUTOTARGETING_TEXT)
        );
        assertThat("добавлен автотаргетинг с апострофом", keywordIDs, hasSize(1));
    }

    @Test
    public void addAutotargetingAndKeywordInSingleRequestTest() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap()
                        .autotargeting(adGroupId)
                        .withBid(MAX_BID)
                        .withContextBid(MAX_BID)
                        .withStrategyPriority(PriorityEnum.NORMAL),
                new KeywordAddMap()
                        .defaultKeyword()
                        .withAdGroupId(adGroupId)
        );
        assertThat("добавлен автотаргетинга и фраза", keywordIDs, hasSize(2));
    }

    @Test
    public void addTwoAutotargetingsInTwoRequestsTest() {
        Long autotargetingId = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap()
                        .autotargeting(adGroupId)
                        .withBid(MAX_BID)
                        .withContextBid(MAX_BID)
                        .withStrategyPriority(PriorityEnum.HIGH)
        ).get(0);
        List<KeywordGetItem> keywordGetItems = api.userSteps.keywordsSteps().keywordsGetById(autotargetingId);

        javaExpectedResult = warnings(
                new Notification(10161,
                        Api5ErrorDetails.FIELD_WILL_NOT_CHANGE_BECAUSE_OF_MANUAL_STRATEGY,
                        capitalize(KeywordAddMap.STRATEGY_PRIORITY)
                ),
                new Notification(7000, Api5ErrorDetails.ONLY_ONE_AUTOTARGETING_IN_ADGROUP)
        );
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(
                        new KeywordAddMap()
                                .shortAutotargeting(adGroupId)
                                .withUserParam1("new param1")
                                .withUserParam2("new param2")
                                .withBid(MIN_BID)
                                .withContextBid(MIN_BID)
                                .withStrategyPriority(PriorityEnum.LOW)
                ),
                javaExpectedResult);

        //DIRECT-152592: cортируем AutotargetingCategories
        List<KeywordGetItem> actualKeywordGetItems = api.userSteps.keywordsSteps().keywordsGetById(autotargetingId);
        assumeThat("", actualKeywordGetItems.size(), equalTo(1));
        actualKeywordGetItems.get(0).getAutotargetingCategories().getValue()
                .getItems().sort(Comparator.comparing(AutotargetingCategory::getCategory));
        assumeThat("", actualKeywordGetItems.size(), equalTo(1));
        keywordGetItems.get(0).getAutotargetingCategories().getValue()
                .getItems().sort(Comparator.comparing(AutotargetingCategory::getCategory));

        assertThat("параметры автотаргетинга не изменились",
                actualKeywordGetItems,
                beanDifferV5(keywordGetItems)
        );
    }

    @Test
    public void addAutotargetingAndKeywordsInSingleRequestTest() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().autotargeting(adGroupId),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw1").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw2").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw3").withBid(MIN_BID)
        );
        assumeThat("добавлен автотаргетинг и три фразы", keywordIDs, hasSize(4));
        Long autotargetingId = keywordIDs.get(0);
        assertThat("у автотаргетинга проставилась правильная ставка по умолчанию",
                api.userSteps.keywordsSteps().keywordsGetById(autotargetingId).get(0).getBid(),
                anyOf(
                        equalTo(DEFAULT_AUTOTARGETING_BID), //perl
                        equalTo(MIN_BID) //java
                )
        );
    }

    @Test
    public void addAutotargetingWithoutBidToAdGroupWithKeywordsTest() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw1").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw2").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw3").withBid(MIN_BID)
        );
        assumeThat("добавлены три фразы", keywordIDs, hasSize(3));
        Long autotargetingId = api.userSteps.keywordsSteps().addShortAutotargeting(adGroupId);
        assertThat("у автотаргетинга проставилась правильная ставка по умолчанию",
                api.userSteps.keywordsSteps().keywordsGetById(autotargetingId).get(0).getBid(),
                equalTo(DEFAULT_AUTOTARGETING_BID)
        );
    }

    @Test
    public void addAutotargetingWithoutBidAndKeywordToAdGroupWithKeywordsTest() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw1").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw2").withBid(MAX_BID)
        );
        assumeThat("добавлены две фразы", keywordIDs, hasSize(2));
        keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().autotargeting(adGroupId),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw3").withBid(MIN_BID)
        );
        assumeThat("добавлен автотаргетинг и фраза", keywordIDs, hasSize(2));
        Long autotargetingId = keywordIDs.get(0);
        assertThat("у автотаргетинга проставилась правильная ставка по умолчанию",
                api.userSteps.keywordsSteps().keywordsGetById(autotargetingId).get(0).getBid(),
                anyOf(
                        equalTo(DEFAULT_AUTOTARGETING_BID), //perl
                        equalTo(MAX_BID) //java
                )
        );
    }

    @Test
    public void addAutotargetingsWithDefaultBidToDifferentAdGroupsTest() {
        Long adGroupId2 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        Long adGroupId3 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().withAdGroupId(adGroupId3).withKeyword("kw1").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId3).withKeyword("kw2").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId3).withKeyword("kw3").withBid(MIN_BID)
        );
        assumeThat("добавлены три фраза", keywordIDs, hasSize(3));
        Long adGroupId4 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().withAdGroupId(adGroupId4).withKeyword("kw1").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId4).withKeyword("kw2").withBid(MAX_BID)
        );
        assumeThat("добавлены две фразы", keywordIDs, hasSize(2));
        Long adGroupId5 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().withAdGroupId(adGroupId5).withKeyword("kw1").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId5).withKeyword("kw2").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId5).withKeyword("kw3").withBid(MIN_BID)
        );
        assumeThat("добавлены три фраза", keywordIDs, hasSize(3));
        keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().autotargeting(adGroupId),
                new KeywordAddMap().autotargeting(adGroupId2),
                new KeywordAddMap().withAdGroupId(adGroupId2).withKeyword("kw1").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId2).withKeyword("kw2").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId2).withKeyword("kw3").withBid(MIN_BID),
                new KeywordAddMap().autotargeting(adGroupId3),
                new KeywordAddMap().autotargeting(adGroupId4),
                new KeywordAddMap().withAdGroupId(adGroupId4).withKeyword("kw3").withBid(MIN_BID),
                new KeywordAddMap().autotargeting(adGroupId5),
                new KeywordAddMap().withAdGroupId(adGroupId5).withKeyword("kw3").withBid(MAX_BID)
        );
        assumeThat("добавлен автотаргетинги и фразы", keywordIDs, hasSize(10));
        assertThat("у всех автотаргетингов проставилась правильная ставка по умолчанию",
                api.userSteps.keywordsSteps().keywordsGetById(keywordIDs.toArray(new Long[keywordIDs.size()])).stream()
                        .filter(item -> KeywordAddMap.AUTOTARGETING_KEYWORD.equals(item.getKeyword()))
                        .map(KeywordGetItem::getBid).collect(Collectors.toList()),
                anyOf(
                        equalTo(Arrays.asList(MIN_BID, DEFAULT_AUTOTARGETING_BID, DEFAULT_AUTOTARGETING_BID,
                                DEFAULT_AUTOTARGETING_BID, DEFAULT_AUTOTARGETING_BID)), //perl
                        equalTo(Arrays.asList(MIN_BID, MIN_BID, DEFAULT_AUTOTARGETING_BID,
                                MAX_BID, DEFAULT_AUTOTARGETING_BID))) //java
        );
    }
}
