package ru.yandex.autotests.directapi.keywords.add;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.general.PriorityEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsBaseRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.features.FeatureNameEnum;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignStrategyAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by adubinkin on 7.02.19.
 * https://st.yandex-team.ru/DIRECT-91581
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Description(
        "Проверка ставок расчитанных при создании автотаргетинга c учетом наличия у клиента фичи 'автотаргетинг на сети'")
@Issue("https://st.yandex-team.ru/DIRECT-91581")
@RunWith(Parameterized.class)
public class AddAutotargetingPricesAutosetTest {

    private static final MoneyCurrency CURRENCY = MoneyCurrency.get(Currency.RUB);
    private static final Long BID1 = 7_000_000L;    // валидная ставка, отличающая от минимальной, максимальной и дефолтной.
    private static final Long BID2 = 2L * BID1;
    private static final Long MIN_PRICE = CURRENCY.getLongMinPrice().longValue();
    private static final Long DEFAULT_PRICE = CURRENCY.getLongDefaultPrice().longValue();
    private static final Long ZERO_PRICE = 0L;

    private static final String CLIENT_WITH_FEATURE = KeywordsLogins.SINGLE_CLIENT_ADD_NEW_RMP_EXT_RM;
    private static final String CLIENT_WITHOUT_FEATURE = Logins.LOGIN_FOR_RUB;

    private static final TextCampaignStrategyAddMap MANUAL_STRATEGY = new TextCampaignStrategyAddMap()
            .withSearch(new TextCampaignSearchStrategyAddMap().defaultHighestPosition())
            .withNetwork(new TextCampaignNetworkStrategyAddMap().defaultMaximumCoverage());

    private static final TextCampaignStrategyAddMap MANUAL_STRATEGY_SEARCH_ONLY = new TextCampaignStrategyAddMap()
            .withSearch(new TextCampaignSearchStrategyAddMap().defaultHighestPosition())
            .withNetwork(new TextCampaignNetworkStrategyAddMap().defaultServingOff());

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);


    @Parameterized.Parameter
    public Boolean hasContextRelevanceMatchFeature;

    @Parameterized.Parameter(1)
    public Boolean withKeywords;

    @Parameterized.Parameter(2)
    public BigDecimal expectedPrice;

    @Parameterized.Parameter(3)
    public BigDecimal expectedCtxPrice;

    @Parameterized.Parameter(4)
    public Boolean strategySearchOnly;


    @Parameterized.Parameters(
            name = "ctxRmFeature={0}, withKeywords={1}, expectedPrice={2}, expectedCtxPrice={3}, strategySearchOnly={4}")
    public static Collection data() {
        return Arrays.asList(new Object[][]{
                {false, false, bidToPrice(MIN_PRICE), bidToPrice(DEFAULT_PRICE), false},
                {false, true, bidToPrice(BID1), bidToPrice(BID2), false},
                {true, false, bidToPrice(MIN_PRICE), bidToPrice(DEFAULT_PRICE), false},
                // *0.8, иначе придется еще раз выставлять Scale
                {true, true, bidToPrice(BID1), bidToPrice(BID2), false},
                {true, false, bidToPrice(MIN_PRICE), bidToPrice(ZERO_PRICE), true},
                {true, true, bidToPrice(BID1), bidToPrice(ZERO_PRICE), true},
        });
    }

    private BidsBaseRecord bidsBaseRecord;
    private TextCampaignStrategyAddMap initialStrategy;

    @Before
    public void prepareData() {
        String login = hasContextRelevanceMatchFeature ? CLIENT_WITH_FEATURE : CLIENT_WITHOUT_FEATURE;
        initialStrategy = strategySearchOnly ? MANUAL_STRATEGY_SEARCH_ONLY : MANUAL_STRATEGY;
        api.as(login);

        Long clientId = Long.valueOf(User.get(login).getClientID());
        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(login);
        jooqDbSteps.featuresSteps().setClientFeature(clientId,
                FeatureNameEnum.CONTEXT_RELEVANCE_MATCH.getValue(), hasContextRelevanceMatchFeature);

        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(login);
        Long campaignId = api.userSteps.campaignSteps().addCampaign(
                new CampaignAddItemMap().defaultCampaignAddItem().withTextCampaign(new TextCampaignAddItemMap()
                        .withBiddingStrategy(initialStrategy)));
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        if (withKeywords) {
            api.userSteps.keywordsSteps().addDefaultKeywordWithBidsAndStrategyPriority(
                    adGroupId, BID1, BID2, PriorityEnum.NORMAL);
        }
        Long autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);

        bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(autotargetingId);
        assumeThat("из таблицы bids_base получена информация о бесфразном таргетинге", bidsBaseRecord, notNullValue());
    }

    @Test
    public void checkPrices() {
        System.out.println(bidsBaseRecord.getPrice() + "--" + bidsBaseRecord.getPriceContext());

        assumeThat("бесфразный таргетинг создался с верной ставкой на поиске: ", bidsBaseRecord.getPrice(),
                equalTo(expectedPrice));
        assertThat("бесфразный таргетинг создался с верной ставкой на сети: ", bidsBaseRecord.getPriceContext(),
                equalTo(expectedCtxPrice));
    }

    private static BigDecimal bidToPrice(Long bid) {
        return BigDecimal.valueOf(bid / 1000000.0).setScale(2);
    }
}
