package ru.yandex.autotests.directapi.keywords.add;


import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.CampaignFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 20.04.2015.
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Description("Проверка у группы времени последнего изменения и статуса синхронизации с БК при добавлении ключевого слова")
@Tag(TagDictionary.TRUNK)
public class AddKeywordsAdGroupLastChangeAndBsSyncedTest {
    private static LogSteps log = LogSteps.getLogger(AddKeywordsAdGroupLastChangeAndBsSyncedTest.class);
    private static final String client = KeywordsLogins.SINGLE_CLIENT_ADD;

    private static Long id;
    private static int shard;
    private static Long campaignID;
    private static Long adID;
    private static Long groupID;
    private static DateTime lastChange = DateTime.now().minusDays(5);
    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepareGroup() {
        shard = api.userSteps.clientFakeSteps().getUserShard(client);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        adID = api.userSteps.adsSteps().addDefaultTextAd(groupID);

        log.info("Промодерируем фейково");
        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignID);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(groupID);
        api.userSteps.bannersFakeSteps().makeBannersModerated(adID);

        log.info("Выставим статусы синхронизации с БК фейково");
        api.userSteps.campaignFakeSteps().setBSSynced(campaignID, true);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(groupID, Status.YES);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adID, Status.YES);

        log.info("Сдвинем время последнего изменения на несколько дней");
        api.userSteps.campaignFakeSteps().setLastChange(campaignID, lastChange.toString());
        api.userSteps.groupFakeSteps().setLastChange(groupID, lastChange.toString());
        api.userSteps.bannersFakeSteps().setLastChange(adID, lastChange.toString());

        log.info("Добавим в группу ключевое слово");
        id = api.userSteps.keywordsSteps().addDefaultKeyword(groupID);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3527")
    public void campaignStatusBsSyncedTest() {
        CampaignFakeInfo campaignInfo = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID);

        assertThat("статус синхронизации с БК у кампании после добавления ключевого слова",
                campaignInfo.getStatusBsSynced(), equalTo(Status.YES));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3528")
    public void bannerStatusBsSyncedTest() {
        BannerFakeInfo bannerInfo = api.userSteps.bannersFakeSteps().getBannerParams(adID);

        assertThat("статус синхронизации с БК у баннера после добавления ключевого слова",
                bannerInfo.getStatusBsSynced(), equalTo(Status.YES));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3529")
    public void groupStatusBsSyncedTest() {
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(groupID);

        assertThat("статус синхронизации с БК у группы после добавления ключевого слова",
                groupInfo.getStatusBsSynced(), equalTo(Status.NO));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3530")
    public void campaignLastChangeTest() {
        CampaignFakeInfo campaignInfo = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID);
        assumeThat("время последнего изменения кампании", campaignInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(campaignInfo.getLastChange());

        assertThat("время последнего изменения кампании после добавления ключевого слов", date, equalTo(lastChange.toLocalDate()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3531")
    public void bannerLastChangeTest() {
        BannerFakeInfo bannerInfo = api.userSteps.bannersFakeSteps().getBannerParams(adID);
        assumeThat("время последнего изменения кампании", bannerInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(bannerInfo.getLastChange());

        assertThat("время последнего изменения баннера после добавления ключевого слова", date, equalTo(lastChange.toLocalDate()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3532")
    public void groupLastChangeTest() {
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(groupID);
        assumeThat("время последнего изменения группы", groupInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(groupInfo.getLastChange());

        assertThat("время последнего изменения группы после добавления ключевого слова", date, equalTo(LocalDate.now()));
    }
}
