package ru.yandex.autotests.directapi.keywords.add;

import java.util.Arrays;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;

/**
 * Created by semkagtn on 16.10.15.
 * https://st.yandex-team.ru/TESTIRT-7489
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Description("Проверка минимального и максимального значения ставок (Bid) для разных валют")
@Issue("https://st.yandex-team.ru/DIRECT-45319")
@RunWith(Parameterized.class)
public class AddKeywordsAndAutotargetingMinAndMaxBidTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter
    public Currency currency;

    @Parameterized.Parameter(1)
    public String stringCurrency;

    @Parameterized.Parameter(2)
    public String login;

    @Parameterized.Parameters(name = "Currency = {0}")
    public static Iterable<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {Currency.RUB, MoneyCurrency.get(Currency.RUB).getNamePluralI(api.locale()), Logins.LOGIN_FOR_RUB},
                {Currency.UAH, MoneyCurrency.get(Currency.UAH).getNameCasusDativus(api.locale()), Logins.LOGIN_FOR_UAH},
                {Currency.KZT, MoneyCurrency.get(Currency.KZT).getNameCasusDativus(api.locale()), Logins.LOGIN_FOR_KZT},
                {Currency.EUR, MoneyCurrency.get(Currency.EUR).getAbbreviation(api.locale()), Logins.LOGIN_FOR_EUR},
                {Currency.USD, MoneyCurrency.get(Currency.USD).getNameCasusDativus(api.locale()), Logins.LOGIN_FOR_USD},
                {Currency.CHF, MoneyCurrency.get(Currency.CHF).getNameCasusDativus(api.locale()), Logins.LOGIN_FOR_CHF},
                {Currency.TRY, MoneyCurrency.get(Currency.TRY).getNamePluralI(api.locale()), Logins.LOGIN_FOR_TRY},
                {Currency.BYN, MoneyCurrency.get(Currency.BYN).getNamePluralI(api.locale()), Logins.CLIENT_BYN}
        });
    }

    private long adGroupId;

    private Long maxBid;
    private Long greaterThanMaxBid;
    private Long minBid;
    private Long lessThanMinBid;

    @Before
    @Step("Подготовка данных для теста")
    public void createAdGroup() {
        api.as(login);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(login);

        long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup((int) campaignId);
        MoneyCurrency moneyCurrency = MoneyCurrency.get(currency);
        maxBid = moneyCurrency.getLongMaxPrice().longValue();
        greaterThanMaxBid = moneyCurrency.getMaxPrice().getNext().bidLong().longValue();
        minBid = moneyCurrency.getLongMinPrice().longValue();
        lessThanMinBid = moneyCurrency.getMinPrice().getPrevious().bidLong().longValue();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3545")
    public void createKeywordWithMinimumBid() {
        api.userSteps.keywordsSteps().shouldGetResultOn(Action.ADD, new AddRequestMap().withKeywords(new KeywordAddMap()
                        .defaultKeyword()
                        .withAdGroupId(adGroupId)
                        .withBid(minBid)),
                ExpectedResult.success());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3546")
    public void createKeywordWithLessThanMinimumBid() {
        api.userSteps.keywordsSteps().shouldGetResultOn(Action.ADD, new AddRequestMap().withKeywords(new KeywordAddMap()
                        .defaultKeyword()
                        .withAdGroupId(adGroupId)
                        .withBid(lessThanMinBid)),
                ExpectedResult
                        .errors(new Notification(5005, Api5ErrorDetails.BID_VALUE_IN_FIELD_MUST_NOT_BE_LESS,
                                capitalize(KeywordAddMap.BID),
                                capitalize(stringCurrency), minBid)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3547")
    public void createKeywordWithMaximumBid() {
        api.userSteps.keywordsSteps().shouldGetResultOn(Action.ADD, new AddRequestMap().withKeywords(new KeywordAddMap()
                        .defaultKeyword()
                        .withAdGroupId(adGroupId)
                        .withBid(maxBid)),
                ExpectedResult.success());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3548")
    public void createKeywordWithGreaterThanMaximumBid() {
        api.userSteps.keywordsSteps().shouldGetResultOn(Action.ADD, new AddRequestMap().withKeywords(new KeywordAddMap()
                        .defaultKeyword()
                        .withAdGroupId(adGroupId)
                        .withBid(greaterThanMaxBid)),
                ExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetails.BID_VALUE_IN_FIELD_MUST_NOT_BE_MORE,
                        capitalize(KeywordAddMap.BID),
                        capitalize(stringCurrency), maxBid)));
    }

    @Test
    public void createAutotargetingWithMinimumBid() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(
                        new KeywordAddMap()
                                .autotargeting(adGroupId)
                                .withBid(minBid)
                ),
                ExpectedResult.success()
        );
    }

    @Test
    public void createAutotargetingWithBidLessThanMinimum() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(
                        new KeywordAddMap()
                                .autotargeting(adGroupId)
                                .withBid(lessThanMinBid)
                ),
                ExpectedResult.errors(
                        new Notification(
                                5005,
                                Api5ErrorDetails.BID_VALUE_IN_FIELD_MUST_NOT_BE_LESS,
                                capitalize(KeywordAddMap.BID),
                                capitalize(stringCurrency),
                                minBid
                        )
                ));
    }

    @Test
    public void createAutotargetingWithMaximumBid() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(
                        new KeywordAddMap()
                                .autotargeting(adGroupId)
                                .withBid(MoneyCurrency.get(currency).getLongMaxPrice().longValue())
                ),
                ExpectedResult.success()
        );
    }


    @Test
    public void createAutotargetingWithBidGreaterThanMaximum() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(
                        new KeywordAddMap()
                                .autotargeting(adGroupId)
                                .withBid(greaterThanMaxBid)
                ),
               ExpectedResult.errors(
                                new Notification(
                                        5005,
                                        Api5ErrorDetails.BID_VALUE_IN_FIELD_MUST_NOT_BE_MORE,
                                        capitalize(KeywordAddMap.BID),
                                        capitalize(stringCurrency),
                                        maxBid
                                )
               ));
    }
}
