package ru.yandex.autotests.directapi.keywords.add;

import java.util.Arrays;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;

/**
 * Created by semkagtn on 16.10.15.
 * https://st.yandex-team.ru/TESTIRT-7489
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Description("Проверка минимального и максимального значения ставок (ContextBid) для разных валют")
@Issue("https://st.yandex-team.ru/DIRECT-45319")
@RunWith(Parameterized.class)
public class AddKeywordsMinAndMaxContextBidTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter
    public Currency currency;

    @Parameterized.Parameter(1)
    public String stringCurrency;

    @Parameterized.Parameter(2)
    public String login;

    @Parameterized.Parameters(name = "Currency = {0}")
    public static Iterable<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {Currency.RUB, MoneyCurrency.get(Currency.RUB).getNamePluralI(api.locale()), Logins.LOGIN_FOR_RUB},
                {Currency.UAH, MoneyCurrency.get(Currency.UAH).getNameCasusDativus(api.locale()), Logins.LOGIN_FOR_UAH},
                {Currency.KZT, MoneyCurrency.get(Currency.KZT).getNameCasusDativus(api.locale()), Logins.LOGIN_FOR_KZT},
                {Currency.EUR, MoneyCurrency.get(Currency.EUR).getAbbreviation(api.locale()), Logins.LOGIN_FOR_EUR},
                {Currency.USD, MoneyCurrency.get(Currency.USD).getNameCasusDativus(api.locale()), Logins.LOGIN_FOR_USD},
                {Currency.CHF, MoneyCurrency.get(Currency.CHF).getNameCasusDativus(api.locale()), Logins.LOGIN_FOR_CHF},
                {Currency.TRY, MoneyCurrency.get(Currency.TRY).getNamePluralI(api.locale()), Logins.LOGIN_FOR_TRY},
                {Currency.BYN, MoneyCurrency.get(Currency.BYN).getNamePluralI(api.locale()), Logins.CLIENT_BYN}
        });
    }

    private long adGroupId;

    @Before
    @Step("Подготовка данных для теста")
    public void createAdGroup() {
        api.as(login);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(login);

        long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup((int) campaignId);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3552")
    public void createKeywordWithMinimumContextBid() {
        long minimumBid = MoneyCurrency.get(currency).getLongMinPrice().longValue();

        api.userSteps.keywordsSteps().shouldGetResultOn(Action.ADD, new AddRequestMap().withKeywords(new KeywordAddMap()
                        .defaultKeyword()
                        .withAdGroupId(adGroupId)
                        .withContextBid(minimumBid)),
                ExpectedResult.success());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3551")
    public void createKeywordWithLessThanMinimumContextBid() {
        Money minBid = MoneyCurrency.get(currency).getMinPrice();
        String javaStringCurrency = stringCurrency;
        if (currency.equals(Currency.EUR)) {
            javaStringCurrency = capitalize(stringCurrency);
        }
        api.userSteps.keywordsSteps().shouldGetResultOn(Action.ADD, new AddRequestMap().withKeywords(new KeywordAddMap()
                        .defaultKeyword()
                        .withAdGroupId(adGroupId)
                        .withContextBid(minBid.getPrevious().bidLong().longValue())),
                ExpectedResult.errors(new Notification(5005,
                                        Api5ErrorDetails.BID_VALUE_IN_FIELD_MUST_NOT_BE_LESS,
                                        capitalize(KeywordAddMap.CONTEXT_BID), javaStringCurrency,
                                        minBid.bidLong().longValue())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3550")
    public void createKeywordWithMaximumContextBid() {
        long maximumBid = MoneyCurrency.get(currency).getLongMaxPrice().longValue();

        api.userSteps.keywordsSteps().shouldGetResultOn(Action.ADD, new AddRequestMap().withKeywords(new KeywordAddMap()
                        .defaultKeyword()
                        .withAdGroupId(adGroupId)
                        .withContextBid(maximumBid)),
                ExpectedResult.success());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3549")
    public void createKeywordWithGreaterThanMaximumBid() {
        Money maxBid = MoneyCurrency.get(currency).getMaxPrice();
        String javaStringCurrency = stringCurrency;
        if (currency.equals(Currency.EUR)) {
            javaStringCurrency = capitalize(stringCurrency);
        }
        api.userSteps.keywordsSteps().shouldGetResultOn(Action.ADD, new AddRequestMap().withKeywords(new KeywordAddMap()
                        .defaultKeyword()
                        .withAdGroupId(adGroupId)
                        .withContextBid(maxBid.getNext().bidLong().longValue())),
                ExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetails.BID_VALUE_IN_FIELD_MUST_NOT_BE_MORE,
                        capitalize(KeywordAddMap.CONTEXT_BID),
                        javaStringCurrency,
                        maxBid.bidLong().longValue())));
    }
}

