package ru.yandex.autotests.directapi.keywords.add;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.PhraseTexts;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * Created by chicos on 17.04.2015.
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Description("Добавление нескольких ключевых слов в одном запросе")
@Tag(TagDictionary.TRUNK)
public class AddKeywordsMultipleTest {
    private static LogSteps log = LogSteps.getLogger(AddKeywordsMultipleTest.class);
    private static final String client = KeywordsLogins.SINGLE_CLIENT_ADD;

    private static Long groupID;
    private static Long campaignID;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepareCampaign(){
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
    }

    @Before
    public void prepareGroup(){
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3553")
    public void addMultipleKeywordsInOneRequestTest() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(
                        new KeywordAddMap()
                                .defaultKeyword()
                                .withAdGroupId(groupID),
                        new KeywordAddMap()
                                .defaultKeyword()
                                .withAdGroupId(groupID)),
                ExpectedResult.success(),
                ExpectedResult.success()
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3554")
    public void addSameKeywordsForDifferentGroupsTest() {
        log.info("Создадим вторую группу");
        long anotherGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        String word = PhraseTexts.getRandom();
        KeywordAddMap item1 = new KeywordAddMap().withKeyword(word).withAdGroupId(groupID);
        KeywordAddMap item2 = new KeywordAddMap().withKeyword(word).withAdGroupId(anotherGroupID);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(item1, item2),
                ExpectedResult.success(),
                ExpectedResult.success()
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3555")
    public void keywordWithSuccessErrorTest() {
        long nonExistingGroup = 9478374l;
        KeywordAddMap item1 = new KeywordAddMap().shortKeyword().withAdGroupId(groupID);
        KeywordAddMap item2 = new KeywordAddMap().shortKeyword().withAdGroupId(nonExistingGroup);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(item1, item2),
                ExpectedResult.success(),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3556")
    public void addSameKeywordsInOneRequestTest() {
        KeywordAddMap item = new KeywordAddMap().shortKeyword().withAdGroupId(groupID);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(item, item),
                ExpectedResult.success(),
                ExpectedResult.warnings(new Notification(10140))
        );
    }

    @Test
    @Description("При добавлении одинаковых ключевых слов валидация не учитывает различия по второстепенным параметрам UserParam")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3557")
    public void addSameKeywordsDifferentUserParamsTest() {
        String word = PhraseTexts.getRandom();
        KeywordAddMap item1 = new KeywordAddMap()
                .withKeyword(word)
                .withUserParam1("param0")
                .withUserParam2("param1")
                .withAdGroupId(groupID);
        KeywordAddMap item2 = new KeywordAddMap()
                .withKeyword(word)
                .withUserParam1("param2")
                .withUserParam2("param3")
                .withAdGroupId(groupID);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(item1, item2),
                ExpectedResult.success(),
                ExpectedResult.warnings(new Notification(10140))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3558")
    public void keywordWithSuccessErrorWarningTest() {
        long nonExistingGroup = 9478374l;
        KeywordAddMap item1 = new KeywordAddMap().shortKeyword().withAdGroupId(groupID);
        KeywordAddMap item2 = new KeywordAddMap().shortKeyword().withAdGroupId(nonExistingGroup);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(item1, item2, item1),
                ExpectedResult.success(),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND)),
                ExpectedResult.warnings(new Notification(10140))
        );
    }
}
