package ru.yandex.autotests.directapi.keywords.add;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.apiclient.version5.ServiceNames;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error.Common.notClientInClientLogin;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 18.04.15
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Stories(ApiStories.UNITS)
@Description("Списание баллов при добавлении ключевых слов")
public class AddKeywordsUnitsTest {
    private static LogSteps log = LogSteps.getLogger(AddKeywordsUnitsTest.class);
    private static final String singleClient = KeywordsLogins.UNITS_ADD_CLIENT;

    private static final String agency = KeywordsLogins.AGENCY_UNITS_ADD;
    private static final String subclient = KeywordsLogins.SUBCLIENT_UNITS_ADD;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int ADD_COST = 20;
    private static final int ITEM_COST_ON_ADD = 2;
    private static final int ADD_FAULT_COST = 20;
    private static final int GENERAL_FAULT_COST = 50;

    private static long groupID1;
    private static long groupID2;

    @BeforeClass
    public static void createGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(singleClient);
        Long clientCampaignID = api.as(singleClient).userSteps.campaignSteps().addDefaultTextCampaign(singleClient);
        groupID1 = api.userSteps.adGroupsSteps().addDefaultGroup(clientCampaignID);

        Long agencyCampaignID = api.as(agency).userSteps.campaignSteps().addDefaultTextCampaign(subclient);
        groupID2 = api.userSteps.adGroupsSteps().addDefaultGroup(agencyCampaignID, subclient);
    }

    @Before
    @Step("Сбросим использованные баллы")
    public void prepareClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(singleClient);
    }

    @Test
    @Description("Добавление одиночного ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3572")
    public void addSingleKeywordTest() {
        api.as(singleClient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.keywordsSteps().addDefaultKeyword(groupID1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assertThat("стоимость добавления одного ключевого слова", unitsBefore - unitsAfter,
                equalTo(ADD_COST + ITEM_COST_ON_ADD));
    }

    @Test
    @Description("Добавление нескольких ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3573")
    public void addSeveralKeywordsTest() {
        api.as(singleClient);
        KeywordAddMap firstKeyword = new KeywordAddMap().shortKeyword().withAdGroupId(groupID1);
        KeywordAddMap secondKeyword = new KeywordAddMap().shortKeyword().withAdGroupId(groupID1);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.keywordsSteps().keywordsAdd(firstKeyword, secondKeyword);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость добавления двух ключевых слов", unitsBefore - unitsAfter,
                equalTo(ADD_COST + ITEM_COST_ON_ADD * 2));
    }

    @Test
    @Description("Добавление одного невалидного ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3571")
    public void addInvalidKeywordItemTest() {
        api.as(singleClient);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(new KeywordAddMap().shortKeyword()),
                ExpectedResult
                        .errors(new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                capitalize(KeywordAddMap.ADGROUP_ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость ошибки при добавлении ключевого слова", unitsBefore - unitsAfter,
                equalTo(ADD_COST + ADD_FAULT_COST));
    }

    @Test
    @Description("Добавление двух невалидных ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3574")
    public void addFewInvalidKeywordsTest() {
        api.as(singleClient);
        KeywordAddMap firstInvalid = new KeywordAddMap().shortKeyword().withAdGroupId(0);
        KeywordAddMap secondInvalid = new KeywordAddMap().withKeyword("!").withAdGroupId(groupID1);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                ServiceNames.KEYWORDS,
                singleClient,
                Action.ADD,
                new AddRequestMap().withKeywords(firstInvalid, secondInvalid),
                ExpectedResult
                        .errors(new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                capitalize(KeywordAddMap.ADGROUP_ID))),
                ExpectedResult
                        .errors(new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "!", "!")));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость ошибки при добавлении двух ключевых слов", unitsBefore - unitsAfter,
                equalTo(ADD_COST + ADD_FAULT_COST * 2));
    }

    @Test
    @Description("Добавление одного корректного и одного невалидного ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3575")
    public void addOneValidOneInvalidKeywordsTest() {
        api.as(singleClient);
        KeywordAddMap firstValid = new KeywordAddMap().shortKeyword().withAdGroupId(groupID1);
        KeywordAddMap secondInvalid = new KeywordAddMap().shortKeyword().withAdGroupId(0);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(firstValid, secondInvalid),
                ExpectedResult.success(),
                ExpectedResult
                        .errors(new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                capitalize(KeywordAddMap.ADGROUP_ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость ошибки при добавлении нескольких ключевых слов",
                unitsBefore - unitsAfter, equalTo(ADD_COST + ADD_FAULT_COST + ITEM_COST_ON_ADD));
    }

    @Test
    @Description("Общая ошибка при добавлении ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3576")
    public void generalErrorOnAddKeywordTest() {
        api.as(singleClient);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.ADD,
                new AddRequestMap().withKeywords(),
                new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                        capitalize(AddRequestMap.KEYWORDS), 1));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость ошибки при добавлении ключевого слова", unitsBefore - unitsAfter,
                equalTo(GENERAL_FAULT_COST));
    }

    @Test
    @Description("Недостаточное колличество баллов при добавлнии ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3577")
    public void notEnoughUnitsOnAddKeywordTest() {
        api.as(singleClient);
        log.info("Обнулим колличество доступных пользователю баллов");
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(singleClient, unitsBefore);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assumeThat("доступные баллы пользователя", unitsBefore, equalTo(0));

        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.ADD,
                new AddRequestMap()
                        .withKeywords(new KeywordAddMap()
                                .shortKeyword()
                                .withAdGroupId(groupID1)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assertThat("доступные баллы пользователя после ошибки", unitsAfter, equalTo(0));
    }

    @Test
    @Description("Добавление агентством нескольких ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3578")
    public void addKeywordsUnitsCostByAgencyTest() {
        api.as(agency);
        KeywordAddMap firstKeyword = new KeywordAddMap().shortKeyword().withAdGroupId(groupID2);
        KeywordAddMap secondKeyword = new KeywordAddMap().shortKeyword().withAdGroupId(groupID2);

        int agencyUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int scUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        api.userSteps.keywordsSteps().keywordsAdd(subclient, firstKeyword, secondKeyword);
        int scUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        int agencyUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);

        assertThat("списание баллов у агентства", agencyUnitsBefore - agencyUnitsAfter, equalTo(0));
        assertThat("списание баллов у субклиента", scUnitsBefore - scUnitsAfter,
                equalTo(ADD_COST + 2 * ITEM_COST_ON_ADD));
    }

    @Test
    @Description("Списание баллов у агентства при вызове метода без указания логина субклиента")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3579")
    public void unknownClientUnitsCostByAgencyTest() {
        api.as(agency);

        int agencyUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.ADD,
                new AddRequestMap().withKeywords(new KeywordAddMap()
                        .defaultKeyword()
                        .withAdGroupId(groupID2)),
                notClientInClientLogin());
        int agencyUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);

        assertThat("списание баллов у агентства", agencyUnitsBefore - agencyUnitsAfter, equalTo(GENERAL_FAULT_COST));
    }
}
