package ru.yandex.autotests.directapi.keywords.add;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsRecord;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by pavryabov on 07.09.15.
 * https://st.yandex-team.ru/TESTIRT-7035
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Description("Проверка ставки после создания. Должна выставлять минимальная ставка для ручной стратегии" +
        "https://st.yandex-team.ru/DIRECT-69182#1518707517000")
@Issue("https://st.yandex-team.ru/DIRECT-45741")
@RunWith(Parameterized.class)
public class CheckPriceAfterAddCurrencyTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public String login;

    @Parameterized.Parameter(1)
    public Currency currency;

    @Parameterized.Parameters(name = "Login = {0}, Currency = {1}")
    public static Collection data() {
        Object[][] data = new Object[][]{
                {Logins.LOGIN_FOR_RUB, Currency.RUB},
                {Logins.LOGIN_FOR_USD, Currency.USD},
                {Logins.LOGIN_FOR_UAH, Currency.UAH},
                {Logins.LOGIN_FOR_EUR, Currency.EUR},
                {Logins.LOGIN_FOR_KZT, Currency.KZT},
                {Logins.LOGIN_FOR_CHF, Currency.CHF},
                {Logins.LOGIN_FOR_TRY, Currency.TRY},
                {Logins.CLIENT_BYN, Currency.BYN}
        };
        return Arrays.asList(data);
    }

    private Long id;
    private int shard;

    @Before
    public void prepareData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(login);
        shard = api.userSteps.clientFakeSteps().getUserShard(login);
        Long campaignID = api.as(login).userSteps.campaignSteps().addDefaultTextCampaign(login);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        id = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3583")
    public void checkPrice() {
        BidsRecord bids = api.userSteps.getDirectJooqDbSteps().useShard(shard).bidsSteps().getBidById(id);
        assumeThat("из таблицы bids получена информация о фразе", bids, notNullValue());
        assertThat("фраза создалась с минимальной ставкой", bids.getPrice().floatValue(),
                equalTo(MoneyCurrency.get(currency).getMinPrice().floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3584")
    public void checkPriceContext() {
        BidsRecord bids = api.userSteps.getDirectJooqDbSteps().useShard(shard).bidsSteps().getBidById(id);
        assumeThat("из таблицы bids получена информация о фразе", bids, notNullValue());
        assertThat("фраза создалась с минимальной ставкой", bids.getPriceContext().floatValue(),
                equalTo(MoneyCurrency.get(currency).getMinPrice().floatValue()));
    }
}
