package ru.yandex.autotests.directapi.keywords.add;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsRecord;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import java.util.Arrays;
import java.util.Collection;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Description("Проверка ставки после создания (https://st.yandex-team.ru/DIRECT-69182#1518707517000)")
@Issue("https://st.yandex-team.ru/DIRECT-45741")
@RunWith(Parameterized.class)
public class CheckPriceAfterAddWithDifferentStrategiesTest {

    private static final String CLIENT = KeywordsLogins.SINGLE_CLIENT_ADD;
    static final Float MIN_PRICE = MoneyCurrency.get(Currency.RUB).getMinPrice().floatValue();

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public TextCampaignSearchStrategyAddMap searchStrategy;

    @Parameterized.Parameter(value = 2)
    public TextCampaignNetworkStrategyAddMap networkStrategy;

    @Parameterized.Parameter(value = 3)
    public Float expectedPrice;

    @Parameterized.Parameter(value = 4)
    public Float expectedContextPrice;

    @Parameterized.Parameters(name = "{0}")
    public static Collection data() {
        Object[][] data = new Object[][]{
                {"Auto + Default",
                        new TextCampaignSearchStrategyAddMap().defaultWbMaximumClicks(Currency.RUB),
                        new TextCampaignNetworkStrategyAddMap().defaultNetworkDefault(),
                        0f, 0f},
                {"Manual + MaximumCoverage",
                        new TextCampaignSearchStrategyAddMap().defaultHighestPosition(),
                        new TextCampaignNetworkStrategyAddMap().defaultMaximumCoverage(),
                        MIN_PRICE, MIN_PRICE},
                {"ServingOff + Auto",
                        new TextCampaignSearchStrategyAddMap().defaultServingOff(),
                        new TextCampaignNetworkStrategyAddMap().defaultAverageCpc(Currency.RUB),
                        0f, 0f},
                {"ServingOff + MaximumCoverage",
                        new TextCampaignSearchStrategyAddMap().defaultServingOff(),
                        new TextCampaignNetworkStrategyAddMap().defaultMaximumCoverage(),
                        0f, MIN_PRICE},
                {"Auto + ServingOff",
                        new TextCampaignSearchStrategyAddMap().defaultAverageCpc(Currency.RUB),
                        new TextCampaignNetworkStrategyAddMap().defaultServingOff(),
                        0f, 0f},
                {"HighestPosition + ServingOff",
                        new TextCampaignSearchStrategyAddMap().defaultHighestPosition(),
                        new TextCampaignNetworkStrategyAddMap().defaultServingOff(),
                        MIN_PRICE, 0f},
        };
        return Arrays.asList(data);
    }

    private Long id;

    @Before
    public void prepareData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        Long campaignID = api.as(CLIENT).userSteps.campaignSteps().addDefaultTextCampaignWithStrategies(
                searchStrategy,
                networkStrategy,
                CLIENT);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        id = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
    }

    @Test
    public void checkPrice() {
        BidsRecord bids = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT).bidsSteps().getBidById(id);
        assumeThat("из таблицы bids получена информация о фразе", bids, notNullValue());
        assertThat("фраза создалась с минимальной ставкой", bids.getPrice().floatValue(),
                equalTo(expectedPrice));
    }

    @Test
    public void checkPriceContext() {
        BidsRecord bids = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT).bidsSteps().getBidById(id);
        assumeThat("из таблицы bids получена информация о фразе", bids, notNullValue());
        assertThat("фраза создалась с минимальной ставкой", bids.getPriceContext().floatValue(),
                equalTo(expectedContextPrice));
    }
}

