package ru.yandex.autotests.directapi.keywords.add;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.yandex.direct.api.v5.general.PriorityEnum;
import com.yandex.direct.api.v5.keywords.KeywordFieldEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordGetItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.anyOf;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by gerdler on 04.04.17.
 * https://st.yandex-team.ru/DIRECT-64666
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Issue("https://st.yandex-team.ru/DIRECT-64386")
@Description("Проверка возвращаемых значений для повторно созданного автотаргетинга")
public class RecreateAutotargetingTest {
    private static final String CLIENT = KeywordsLogins.SINGLE_CLIENT_ADD;
    private static final MoneyCurrency CURRENCY_RUB = MoneyCurrency.get(Currency.RUB);
    private static final Long MIN_BID = CURRENCY_RUB.getLongMinPrice().longValue();
    private static final Long DEFAULT_BID = CURRENCY_RUB.getLongDefaultPrice().longValue();
    private static final Long BID1 = MIN_BID * 2L;
    private static final Long BID2 = BID1 * 2L;
    private static final Long MAX_BID = CURRENCY_RUB.getLongMaxPrice().longValue();

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long campaignId;
    private Long adGroupId;
    private static Long autoTargetingKeywordId;
    private static final List<KeywordFieldEnum> FIELDS_TO_CHECK = Arrays.asList(
            KeywordFieldEnum.BID,
            KeywordFieldEnum.CONTEXT_BID,
            KeywordFieldEnum.STRATEGY_PRIORITY,
            KeywordFieldEnum.USER_PARAM_1,
            KeywordFieldEnum.USER_PARAM_2
    );

    @BeforeClass
    public static void prepareCampaign() {
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
    }

    @Before
    public void prepareDeletedAutoTargeting() {
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        autoTargetingKeywordId = api.userSteps.keywordsSteps()
                .addAutotargetingWithBidsAndStrategyPriority(adGroupId, BID1, BID2, PriorityEnum.HIGH);
        api.userSteps.keywordsSteps().keywordsDelete(autoTargetingKeywordId);
    }

    @Test
    public void recreateAutoTargetingWithDefaultBids() {
        assumeThat("автотаргетинг повторно создался с темже идентификатором",
                api.userSteps.keywordsSteps().addShortAutotargeting(adGroupId),
                equalTo(autoTargetingKeywordId)
        );

        assertThat("получили один автотаргетинг с правильными значениями полей",
                api.userSteps.keywordsSteps().keywordsGetByIdWithFields(FIELDS_TO_CHECK, autoTargetingKeywordId),
                beanDifferV5(
                        Arrays.asList(
                                (KeywordGetItem) new KeywordGetItemMap()
                                        .withBid(MIN_BID)
                                        .withContextBid(DEFAULT_BID)
                                        .withStrategyPriority(PriorityEnum.NORMAL)
                                        .withUserParam1(null)
                                        .withUserParam2(null)
                                        .getBean()
                        )
                )
        );
    }

    @Test
    public void recreateAutotargetingWithDefaultBidsInAdGroupWithKeywords() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw1").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw2").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw3").withBid(MIN_BID)
        );
        assumeThat("добавлены три фразы", keywordIDs, hasSize(3));
        assumeThat("автотаргетинг повторно создался с темже идентификатором",
                api.userSteps.keywordsSteps().addShortAutotargeting(adGroupId),
                equalTo(autoTargetingKeywordId)
        );

        assertThat("получили один автотаргетинг с правильными значениями полей",
                api.userSteps.keywordsSteps().keywordsGetByIdWithFields(FIELDS_TO_CHECK, autoTargetingKeywordId),
                beanDifferV5(
                        Arrays.asList(
                                (KeywordGetItem) new KeywordGetItemMap()
                                        .withBid(15000200000L)
                                        .withContextBid(MIN_BID)
                                        .withStrategyPriority(PriorityEnum.NORMAL)
                                        .withUserParam1(null)
                                        .withUserParam2(null)
                                        .getBean()
                        )
                )
        );
    }

    @Test
    public void recreateAutotargetingWithDefaultBidsAndAddKeyrowdsInAdGroupWithKeywords() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw1").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw2").withBid(MAX_BID)
        );
        assumeThat("добавлены две фразы", keywordIDs, hasSize(2));
        keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().shortAutotargeting(adGroupId),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw3").withBid(MIN_BID)
        );
        assumeThat("добавлен автотаргетинг и фраза", keywordIDs, hasSize(2));
        assumeThat("автотаргетинг повторно создался с темже идентификатором",
                keywordIDs.get(0),
                equalTo(autoTargetingKeywordId)
        );

        assertThat("получили один автотаргетинг с правильными значениями полей",
                api.userSteps.keywordsSteps().keywordsGetByIdWithFields(FIELDS_TO_CHECK, autoTargetingKeywordId),
                anyOf(beanDifferV5( //для perl
                        Collections.singletonList(
                                (KeywordGetItem) new KeywordGetItemMap()
                                        .withBid(15000200000L)
                                        .withContextBid(MIN_BID)
                                        .withStrategyPriority(PriorityEnum.NORMAL)
                                        .withUserParam1(null)
                                        .withUserParam2(null)
                                        .getBean())
                        ),
                        beanDifferV5( //для java
                                Collections.singletonList(
                                        (KeywordGetItem) new KeywordGetItemMap()
                                                .withBid(MAX_BID)
                                                .withContextBid(MIN_BID)
                                                .withStrategyPriority(PriorityEnum.NORMAL)
                                                .withUserParam1(null)
                                                .withUserParam2(null)
                                                .getBean()
                                )
                        )
                )
        );
    }

    @Test
    public void recreateAutotargetingWithDefaultBidsAndAddKeyrowds() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsAdd(
                new KeywordAddMap().shortAutotargeting(adGroupId),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw1").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw2").withBid(MAX_BID),
                new KeywordAddMap().withAdGroupId(adGroupId).withKeyword("kw3").withBid(MIN_BID)
        );
        assumeThat("добавлен автотаргетинг и три фразы", keywordIDs, hasSize(4));
        assumeThat("автотаргетинг повторно создался с темже идентификатором",
                keywordIDs.get(0),
                equalTo(autoTargetingKeywordId)
        );

        assertThat("получили один автотаргетинг с правильными значениями полей",
                api.userSteps.keywordsSteps().keywordsGetByIdWithFields(FIELDS_TO_CHECK, autoTargetingKeywordId),
                anyOf(beanDifferV5( // для perl
                        Collections.singletonList(
                                (KeywordGetItem) new KeywordGetItemMap()
                                        .withBid(15000200000L)
                                        .withContextBid(MIN_BID)
                                        .withStrategyPriority(PriorityEnum.NORMAL)
                                        .withUserParam1(null)
                                        .withUserParam2(null)
                                        .getBean()
                        )
                ), beanDifferV5( // для java
                        Collections.singletonList(
                                (KeywordGetItem) new KeywordGetItemMap()
                                        .withBid(MIN_BID)
                                        .withContextBid(MIN_BID)
                                        .withStrategyPriority(PriorityEnum.NORMAL)
                                        .withUserParam1(null)
                                        .withUserParam2(null)
                                        .getBean()
                        )
                ))
        );
    }

    @Test
    public void recreateAutoTargetingWithManualBids() {
        assumeThat("автотаргетинг повторно создался с темже идентификатором",
                api.userSteps.keywordsSteps()
                        .addAutotargetingWithBidsAndStrategyPriority(adGroupId, BID2, BID1, PriorityEnum.LOW),
                equalTo(autoTargetingKeywordId)
        );
        assertThat("получили один автоаргетинг с правильными значениями полей",
                api.userSteps.keywordsSteps().keywordsGetByIdWithFields(FIELDS_TO_CHECK, autoTargetingKeywordId),
               anyOf(beanDifferV5( // для perl
                        Collections.singletonList(
                                (KeywordGetItem) new KeywordGetItemMap()
                                        .withBid(BID2)
                                        .withContextBid(BID1)
                                        .withStrategyPriority(PriorityEnum.LOW)
                                        .withUserParam1("123456")
                                        .withUserParam2("superParam")
                                        .getBean()
                        )
                ), beanDifferV5( // для java
                       Collections.singletonList(
                               (KeywordGetItem) new KeywordGetItemMap()
                                       .withBid(BID2)
                                       .withContextBid(MIN_BID)
                                       .withStrategyPriority(PriorityEnum.LOW)
                                       .withUserParam1("123456")
                                       .withUserParam2("superParam")
                                       .getBean()
                       )
               ))
        );
    }
}
