package ru.yandex.autotests.directapi.keywords.add.keyword;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.keywords.KeywordFieldEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordGetItemMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by pavryabov on 25.02.16.
 * https://st.yandex-team.ru/TESTIRT-8621
 */
@Aqua.Test
@Features(KeywordsFeatures.ADD)
@Description("Добавление нескольких ключевых слов в одном запросе")
@Issue("https://st.yandex-team.ru/DIRECT-51237")
@RunWith(Parameterized.class)
public class AddSeveralKeywordsTest {

    private static final String client = KeywordsLogins.SINGLE_CLIENT_ADD;

    private static Long campaignId;

    private Long adGroupId;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public List<String> keywords;

    @Parameterized.Parameter(value = 2)
    public ExpectedResult[] expectedResult;

    @Parameterized.Parameter(value = 3)
    public List<String> expectedKeywords;

    @Parameterized.Parameters(name = "{0}")
    public static Collection fields() {
        Object[][] data = new Object[][]{
                {"Две фразы в кавычках, различающиеся только символом + перед стоп-словом",
                        Arrays.asList("\"слово за слово\"", "\"слово +за слово\""),
                        new ExpectedResult[]{
                                ExpectedResult.success(),
                                ExpectedResult.warnings(new Notification(10140))},
                        Collections.singletonList("\"слово за слово\"")},
                {"Две фразы различающиеся только символом + перед стоп-словом",
                        Arrays.asList("слово за слово", "слово +за слово"),
                        new ExpectedResult[]{
                                ExpectedResult.success(),
                                ExpectedResult.success()},
                        Arrays.asList("слово за слово", "слово +за слово")},
                {"Две фразы, чьи нормальные формы совпадают",
                        Arrays.asList("тур в Ватикан из Москвы", "туры в Ватикан из Москвы"),
                        new ExpectedResult[]{
                                ExpectedResult.success(),
                                ExpectedResult.warnings(new Notification(10140))},
                        Collections.singletonList("тур в Ватикан из Москвы")},
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void createCampaign() {
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
    }

    @Before
    public void prepareGroup() {
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Test
    public void keywordsTest() {
        KeywordAddMap[] keywordAddMaps = keywords.stream()
                .map(keyword -> new KeywordAddMap().withAdGroupId(adGroupId).withKeyword(keyword))
                .toArray(KeywordAddMap[]::new);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(keywordAddMaps),
                expectedResult);
        List<KeywordGetItem> actualKeywords = api.userSteps.keywordsSteps().keywordsGet(
                Arrays.asList(KeywordFieldEnum.KEYWORD),
                new KeywordsSelectionCriteriaMap().withAdGroupIds(adGroupId)
        );
        List<KeywordGetItem> expectedKeywordGetItems = expectedKeywords.stream()
                .map(keyword -> (KeywordGetItem) new KeywordGetItemMap().withKeyword(keyword).getBean())
                .collect(Collectors.toList());
        assertThat("ключевые слова сохранились правильно", actualKeywords, beanDifferV5(expectedKeywordGetItems));
    }
}
