package ru.yandex.autotests.directapi.keywords.add.limit;

import org.apache.commons.lang3.RandomStringUtils;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.version5.ServiceNames;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * Created by buhter on 02.02.2016.
 * https://st.yandex-team.ru/TESTIRT-8383
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-49216")
@Features(KeywordsFeatures.ADD)
@Description("Проверка лимита количества Keywords на одну группу")
public class AddKeywordsGroupLimitTest {
    private static final String CLIENT = Logins.SINGLE_CLIENT_CHANGED_LIMIT;
    private static final int KEYWORDS_IN_GROUP_LIMIT = 10;
    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);
    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();
    private static LogSteps log = LogSteps.getLogger(AddKeywordsGroupLimitTest.class);
    private static Long campaignID;
    @Rule
    public Trashman trashman = new Trashman(api);
    private Long groupID;

    @BeforeClass
    public static void prepareCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT).clientsLimitsSteps()
                .addOrUpdateClientLimitsKeywordCount(
                        Long.valueOf(User.get(CLIENT).getClientID()),
                        (long) KEYWORDS_IN_GROUP_LIMIT);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void init() {
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
    }

    @Test
    public void addMaxKeywordsToGroupTest() {
        String wordToAdd;
        KeywordAddMap[] keywordAddMaps = new KeywordAddMap[KEYWORDS_IN_GROUP_LIMIT];
        ExpectedResult[] results = new ExpectedResult[KEYWORDS_IN_GROUP_LIMIT];
        for (int i = 0; i < KEYWORDS_IN_GROUP_LIMIT; i++) {
            wordToAdd = RandomStringUtils.randomAlphabetic(15);
            keywordAddMaps[i] = new KeywordAddMap()
                    .withKeyword(wordToAdd)
                    .withAdGroupId(groupID);
            results[i] = ExpectedResult.success();
        }

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(keywordAddMaps),
                results
        );
    }

    @Test
    public void addMoreThanMaxKeywordsToGroupTest() {
        String wordToAdd;
        KeywordAddMap[] keywordAddMaps = new KeywordAddMap[KEYWORDS_IN_GROUP_LIMIT + 1];
        ExpectedResult[] results = new ExpectedResult[KEYWORDS_IN_GROUP_LIMIT + 1];
        for (int i = 0; i < KEYWORDS_IN_GROUP_LIMIT + 1; i++) {
            wordToAdd = RandomStringUtils.randomAlphabetic(15);
            keywordAddMaps[i] = new KeywordAddMap()
                    .withKeyword(wordToAdd)
                    .withAdGroupId(groupID);
            results[i] = ExpectedResult.errors(new Notification(7001
                    , Api5ErrorDetails.ADGROUP_COULD_CONTAIN_KEYWORD_COUNT
                    , KEYWORDS_IN_GROUP_LIMIT));

        }

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(keywordAddMaps),
                results
        );
    }

    @Test
    public void addMoreThanMaxSameKeywordsToGroupTest() {
        String wordToAdd = RandomStringUtils.randomAlphabetic(15);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(
                        new KeywordAddMap()
                                .withKeyword(wordToAdd)
                                .withAdGroupId(groupID)),
                ExpectedResult.success()
        );

        KeywordAddMap[] keywordAddMaps = new KeywordAddMap[KEYWORDS_IN_GROUP_LIMIT];
        for (int i = 0; i < KEYWORDS_IN_GROUP_LIMIT; i++) {
            keywordAddMaps[i] = new KeywordAddMap()
                    .withKeyword(wordToAdd)
                    .withAdGroupId(groupID);
        }

        JavaOrPerlExpectedResult[] javaOrPerlExpectedResults = new JavaOrPerlExpectedResult[KEYWORDS_IN_GROUP_LIMIT];
        for (int i = 0; i < KEYWORDS_IN_GROUP_LIMIT; i++) {
            javaOrPerlExpectedResults[i] = JavaOrPerlExpectedResult
                    .java(ExpectedResult.warnings(new Notification(10140)))
                    .perl(ExpectedResult.errors(new Notification(7001
                            , Api5ErrorDetails.ADGROUP_COULD_CONTAIN_KEYWORD_COUNT
                            , KEYWORDS_IN_GROUP_LIMIT)));
        }

        api.userSteps.keywordsSteps().shouldGetResultOn(
                ServiceNames.KEYWORDS,
                CLIENT,
                Action.ADD,
                new AddRequestMap().withKeywords(keywordAddMaps),
                javaOrPerlExpectedResults
        );

    }

    @Test
    public void addKeywordsToGroupWithMaxKeywordsAmountTest() {
        String wordToAdd;
        KeywordAddMap[] keywordAddMaps = new KeywordAddMap[KEYWORDS_IN_GROUP_LIMIT];
        ExpectedResult[] results = new ExpectedResult[KEYWORDS_IN_GROUP_LIMIT];
        for (int i = 0; i < KEYWORDS_IN_GROUP_LIMIT; i++) {
            wordToAdd = RandomStringUtils.randomAlphabetic(15);
            keywordAddMaps[i] = new KeywordAddMap()
                    .withKeyword(wordToAdd)
                    .withAdGroupId(groupID);
            results[i] = ExpectedResult.success();
        }

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(keywordAddMaps),
                results
        );

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(
                        new KeywordAddMap()
                                .withAdGroupId(groupID)
                                .withKeyword(RandomStringUtils.randomAlphabetic(15))
                ),
                ExpectedResult.errors(new Notification(7001, Api5ErrorDetails.ADGROUP_COULD_CONTAIN_KEYWORD_COUNT
                        , KEYWORDS_IN_GROUP_LIMIT))
        );
    }


    @Test
    public void addKeywordsToGroupWithKeywordsMoreThenMaxInTotalTest() {
        String wordToAdd;
        KeywordAddMap[] keywordAddMaps = new KeywordAddMap[KEYWORDS_IN_GROUP_LIMIT - 4];
        ExpectedResult[] results = new ExpectedResult[KEYWORDS_IN_GROUP_LIMIT - 4];
        for (int i = 0; i < KEYWORDS_IN_GROUP_LIMIT - 4; i++) {
            wordToAdd = RandomStringUtils.randomAlphabetic(15);
            keywordAddMaps[i] = new KeywordAddMap()
                    .withKeyword(wordToAdd)
                    .withAdGroupId(groupID);
            results[i] = ExpectedResult.success();
        }

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(keywordAddMaps),
                results
        );

        keywordAddMaps = new KeywordAddMap[5];
        results = new ExpectedResult[5];
        for (int i = 0; i < 5; i++) {
            wordToAdd = RandomStringUtils.randomAlphabetic(15);
            keywordAddMaps[i] = new KeywordAddMap()
                    .withKeyword(wordToAdd)
                    .withAdGroupId(groupID);
            results[i] = ExpectedResult.errors(new Notification(7001
                    , Api5ErrorDetails.ADGROUP_COULD_CONTAIN_KEYWORD_COUNT
                    , KEYWORDS_IN_GROUP_LIMIT));
        }

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(keywordAddMaps)
                , results
        );
    }

    @Test
    public void addKeywordsToGroupWithKeywordsEqualToMaxInTotalTest() {
        String wordToAdd;
        KeywordAddMap[] keywordAddMaps = new KeywordAddMap[KEYWORDS_IN_GROUP_LIMIT - 5];
        ExpectedResult[] results = new ExpectedResult[KEYWORDS_IN_GROUP_LIMIT - 5];
        for (int i = 0; i < KEYWORDS_IN_GROUP_LIMIT - 5; i++) {
            wordToAdd = RandomStringUtils.randomAlphabetic(15);
            keywordAddMaps[i] = new KeywordAddMap()
                    .withKeyword(wordToAdd)
                    .withAdGroupId(groupID);
            results[i] = ExpectedResult.success();
        }

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(keywordAddMaps),
                results
        );

        keywordAddMaps = new KeywordAddMap[5];
        results = new ExpectedResult[5];
        for (int i = 0; i < 5; i++) {
            wordToAdd = RandomStringUtils.randomAlphabetic(15);
            keywordAddMaps[i] = new KeywordAddMap()
                    .withKeyword(wordToAdd)
                    .withAdGroupId(groupID);
            results[i] = ExpectedResult.success();
        }

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap().withKeywords(keywordAddMaps)
                , results
        );
    }
}
