package ru.yandex.autotests.directapi.keywords.delete;

import java.sql.Timestamp;
import java.time.LocalDateTime;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.beans.autotargeting.AutotargetingOpts;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersPhoneflag;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusbssynced;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusshow;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatussitelinksmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusbssynced;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PhrasesStatusbssynced;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PhrasesStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PhrasesStatuspostmoderate;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by gerdler on 19.04.17.
 * https://st.yandex-team.ru/DIRECT-64666
 */
@Aqua.Test
@Features(KeywordsFeatures.DELETE)
@Description("Проверка у объектов даты последнего изменения и статуса синхронизации с БК при удалении автотаргетинга")
@Issue("https://st.yandex-team.ru/DIRECT-64386")
public class DeleteAutotargetingAdGroupLastChangeAndBsSyncedTest {

    private static LogSteps log = LogSteps.getLogger(DeleteAutotargetingAdGroupLastChangeAndBsSyncedTest.class);
    private static final String CLIENT = KeywordsLogins.SINGLE_CLIENT_DELETE;
    private static final Timestamp YESTERDAY = Timestamp.valueOf(LocalDateTime.now().minusDays(1).withNano(0));

    private static Long campaignId;
    private static Long adId;
    private static Long adGroupId;
    private static Long autotargetingId;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @Rule
    public Trashman trashman = new Trashman(api);

    private static DirectJooqDbSteps jooqDbSteps;

    @BeforeClass
    public static void prepareData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        adId = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);

        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT);

        log.info("Проставляем необходимые статусы и дату последнего изменения у объектов");
        jooqDbSteps.campaignsSteps().updateCampaigns(
                jooqDbSteps.campaignsSteps().getCampaignById(campaignId)
                        .setStatusmoderate(CampaignsStatusmoderate.Yes)
                        .setStatusbssynced(CampaignsStatusbssynced.Yes)
                        .setLastchange(YESTERDAY)
        );
        jooqDbSteps.adGroupsSteps().updatePhrases(
                jooqDbSteps.adGroupsSteps().getPhrases(adGroupId)
                        .setStatusmoderate(PhrasesStatusmoderate.Yes)
                        .setStatuspostmoderate(PhrasesStatuspostmoderate.Yes)
                        .setStatusbssynced(PhrasesStatusbssynced.Yes)
                        .setLastchange(YESTERDAY)
        );
        jooqDbSteps.bannersSteps().updateBanners(
                jooqDbSteps.bannersSteps().getBanner(adId)
                        .setStatusshow(BannersStatusshow.Yes)
                        .setStatusmoderate(BannersStatusmoderate.Yes)
                        .setStatuspostmoderate(BannersStatuspostmoderate.Yes)
                        .setPhoneflag(BannersPhoneflag.Yes)
                        .setStatussitelinksmoderate(BannersStatussitelinksmoderate.Yes)
                        .setStatusbssynced(BannersStatusbssynced.Yes)
                        .setLastchange(YESTERDAY)
        );
        jooqDbSteps.bidsBaseSteps().updateBidsBase(
                jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(autotargetingId)
                        .setLastchange(YESTERDAY)
        );

        log.info("Удаляем автотаргетинг");
        api.userSteps.keywordsSteps().keywordsDelete(autotargetingId);
        assumeThat("автотаргетинг удален", jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(autotargetingId).getOpts(), equalTo(AutotargetingOpts.DELETED.toString()));
    }

    @Test
    public void campaignStatusBsSyncedTest() {
        assertThat("статус синхронизации с БК у кампании после удаления автотаргетинга",
                jooqDbSteps.campaignsSteps().getCampaignById(campaignId).getStatusbssynced(),
                equalTo(CampaignsStatusbssynced.Yes)
        );
    }

    @Test
    public void bannerStatusBsSyncedTest() {
        assertThat("статус синхронизации с БК у баннера после удаления автотаргетинга",
                jooqDbSteps.bannersSteps().getBanner(adId).getStatusbssynced(),
                equalTo(BannersStatusbssynced.Yes)
        );
    }

    @Test
    public void groupStatusBsSyncedTest() {
        assertThat("статус синхронизации с БК у группы после удаления автотаргетинга",
                jooqDbSteps.adGroupsSteps().getPhrases(adGroupId).getStatusbssynced(),
                equalTo(PhrasesStatusbssynced.No)
        );
    }

    @Test
    public void campaignLastChangeTest() {
        assertThat("время последнего изменения кампании после удаления автотаргетинга не изменилось",
                jooqDbSteps.campaignsSteps().getCampaignById(campaignId).getLastchange(),
                equalTo(YESTERDAY)
        );
    }

    @Test
    public void bannerLastChangeTest() {
        assertThat("время последнего изменения баннера после удаления автотаргетинга не изменилось",
                jooqDbSteps.bannersSteps().getBanner(adId).getLastchange(),
                equalTo(YESTERDAY)
        );
    }

    @Test
    public void groupLastChangeTest() {
        assertThat("время последнего изменения группы после удаления автотаргетинга не изменилось",
                jooqDbSteps.adGroupsSteps().getPhrases(adGroupId).getLastchange(),
                greaterThan(YESTERDAY)
        );
    }
}
