package ru.yandex.autotests.directapi.keywords.delete;

import java.util.List;

import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.ApiFeatures;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.DeleteRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.extract;
import static ch.lambdaj.Lambda.on;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by semkagtn on 2/27/15.
 * https://st.yandex-team.ru/TESTIRT-4389
 */
@Aqua.Test
@Features(ApiFeatures.DEFECT)
@Issue("https://st.yandex-team.ru/DIRECT-36384")
@Description("Одинаковые идетификаторы в запросе на удаление ключевых слов")
public class DeleteSameIdsTest {

    private static final String LOGIN = KeywordsLogins.SINGLE_CLIENT_DELETE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private long firstKeywordID;
    private long secondKeywordID;

    @Before
    @Step("Подготовка данных для теста")
    public void addUnitsToUser() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        long adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        List<Long> keywordIDs =api.userSteps.keywordsSteps().addDefaultKeywords(adGroupID, 2);
        firstKeywordID = keywordIDs.get(0);
        secondKeywordID = keywordIDs.get(1);
    }

    @Test
    @Description("Два идентификатора в запросе: один продублированный, другой - нет")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3606")
    public void sameIdsInRequest() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(firstKeywordID, firstKeywordID, secondKeywordID)),
                ExpectedResult.errors(new Notification(9800, Api5ErrorDetails.KEYWORD_ID_COULD_NOT_REPEAT_IN_REQUEST)),
                ExpectedResult.errors(new Notification(9800, Api5ErrorDetails.KEYWORD_ID_COULD_NOT_REPEAT_IN_REQUEST)),
                ExpectedResult.success(secondKeywordID));

        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGetById(firstKeywordID, secondKeywordID);
        List<Long> keywordIds = extract(keywords, on(KeywordGetItem.class).getId());
        assertThat("удалился только идентификатор, котороый не был продублирован в запросе",
                keywordIds, contains(equalTo(firstKeywordID)));
    }
}
