package ru.yandex.autotests.directapi.keywords.delete;

import java.util.List;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.DeleteRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 12.03.15.
 * https://st.yandex-team.ru/TESTIRT-4389
 */
@Aqua.Test
@Features(KeywordsFeatures.DELETE)
@Stories(ApiStories.UNITS)
@Issue("https://st.yandex-team.ru/DIRECT-36384")
@Description("Проверка списания баллов при удалении ключевых слов")
public class DeleteUnitsTest {

    private static final String LOGIN = KeywordsLogins.UNITS_DELETE_CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int CALL_COST = 10;
    private static final int KEYWORD_COST = 1;
    private static final int ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR_COST = 50;

    private static long adGroupID;

    @BeforeClass
    public static void createAdGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
    }

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    @Description("Удаление одного ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3612")
    public void deleteOneKeyword() {
        long keywordID = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().keywordsDelete(keywordID);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + KEYWORD_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Удаление двух ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3613")
    public void deleteTwoKeywords() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().addDefaultKeywords(adGroupID, 2);
        long firstKeywordID = keywordIDs.get(0);
        long secondKeywordID = keywordIDs.get(1);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().keywordsDelete(firstKeywordID, secondKeywordID);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + KEYWORD_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Удаление одного невалидного ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3614")
    public void deleteOneInvalidKeyword() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(1L)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.KEYWORD_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Удаление двух невалидных ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3615")
    public void deleteTwoInvalidKeywords() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(1L, 2L)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.KEYWORD_NOT_FOUND)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.KEYWORD_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Удаление одного валидного и одного невалидного ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3616")
    public void deleteOneValidAndOneInvalidKeywords() {
        long keywordID = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(keywordID, 1L)),
                ExpectedResult.success(keywordID),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.KEYWORD_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST + KEYWORD_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Невалидный запрос на удаление ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3617")
    public void invalidRequest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.DELETE,
                new DeleteRequestMap()
                        .withSelectionCriteria(null),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        capitalize(DeleteRequestMap.SELECTION_CRITERIA)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = COMMON_REQUEST_ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Удаление ключевого слова, когда у клиента нету баллов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3618")
    public void deleteKeywordWhenClientHasZeroUnits() {
        long keywordID = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);

        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("у клиента нету баллов", units, equalTo(0));

        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.DELETE,
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(keywordID)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
    }
}
