package ru.yandex.autotests.directapi.keywords.get;

import java.util.Comparator;
import java.util.List;

import com.yandex.direct.api.v5.general.AutotargetingCategoriesEnum;
import com.yandex.direct.api.v5.general.AutotargetingCategory;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsBaseRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.general.AutotargetingCategoryMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordGetItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanEquivalentV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(KeywordsFeatures.GET)
@Description("Проверка нового поля AutotargetingCategories в методе Keywords.get")
@Issue("https://st.yandex-team.ru/DIRECT-149649")
public class GetAutotargetingCategoriesTest {

    private static final String CLIENT = KeywordsLogins.SINGLE_CLIENT_ADD;

    private static final Comparator<AutotargetingCategory> AUTOTARGETING_CATEGORY_COMPARATOR =
            Comparator.comparing(c -> c.getCategory().value());

    private static Long campaignId;
    private static Long adGroupId;
    private static DirectJooqDbSteps jooqDbSteps;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepareCampaign() {
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT);
    }

    @Before
    public void prepareAdGroup() {
        api.protocol(ProtocolType.JSON);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Test
    public void emptyRelevanceMatchCategoriesInDB() {
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
        ).get(0);

        BidsBaseRecord bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(id);
        bidsBaseRecord.setRelevanceMatchCategories("");
        jooqDbSteps.bidsBaseSteps().updateBidsBase(bidsBaseRecord);

        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGetById(id);
        assumeThat("получено ключевое слово", keywords, hasSize(1));

        keywords.get(0).getAutotargetingCategories().getValue().getItems().sort(AUTOTARGETING_CATEGORY_COMPARATOR);

        KeywordGetItemMap expectedKeyword = new KeywordGetItemMap()
                .withId(id)
                .withAutotargetingCategories(new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.ACCESSORY)
                                .withValue(YesNoEnum.YES),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.ALTERNATIVE)
                                .withValue(YesNoEnum.YES),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.BROADER)
                                .withValue(YesNoEnum.YES),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.COMPETITOR)
                                .withValue(YesNoEnum.YES),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.EXACT)
                                .withValue(YesNoEnum.YES));
        assertThat("получены корректные категории автотаргетинга", keywords.get(0),
                beanEquivalentV5(expectedKeyword.getBean()));
    }

    @Test
    public void fullRelevanceMatchCategoriesInDB() {
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
                .withAutotargetingCategories()
        ).get(0);

        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGetById(id);
        assumeThat("получено ключевое слово", keywords, hasSize(1));

        keywords.get(0).getAutotargetingCategories().getValue().getItems().sort(AUTOTARGETING_CATEGORY_COMPARATOR);

        KeywordGetItemMap expectedKeyword = new KeywordGetItemMap()
                .withId(id)
                .withAutotargetingCategories(new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.ACCESSORY)
                                .withValue(YesNoEnum.YES),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.ALTERNATIVE)
                                .withValue(YesNoEnum.YES),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.BROADER)
                                .withValue(YesNoEnum.YES),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.COMPETITOR)
                                .withValue(YesNoEnum.YES),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.EXACT)
                                .withValue(YesNoEnum.YES));
        assertThat("получены корректные категории автотаргетинга", keywords.get(0),
                beanEquivalentV5(expectedKeyword.getBean()));
    }

    @Test
    public void noFullRelevanceMatchCategoriesInDB() {
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
                .withAutotargetingCategories(new AutotargetingCategoryMap()
                        .withCategory(AutotargetingCategoriesEnum.ACCESSORY)
                        .withValue(YesNoEnum.NO))
        ).get(0);

        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGetById(id);
        assumeThat("получено ключевое слово", keywords, hasSize(1));

        keywords.get(0).getAutotargetingCategories().getValue().getItems().sort(AUTOTARGETING_CATEGORY_COMPARATOR);

        KeywordGetItemMap expectedKeyword = new KeywordGetItemMap()
                .withId(id)
                .withAutotargetingCategories(new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.ACCESSORY)
                                .withValue(YesNoEnum.NO),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.ALTERNATIVE)
                                .withValue(YesNoEnum.YES),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.BROADER)
                                .withValue(YesNoEnum.YES),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.COMPETITOR)
                                .withValue(YesNoEnum.YES),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.EXACT)
                                .withValue(YesNoEnum.YES));
        assertThat("получены корректные категории автотаргетинга", keywords.get(0),
                beanEquivalentV5(expectedKeyword.getBean()));
    }

    @Test
    public void autotargetingCategoriesAndInvalidTypeJSON() {
        Long mobileContentCampaignId = api.userSteps.campaignSteps().addDefaultMobileAppCampaign(CLIENT);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultMobileGroup(mobileContentCampaignId);
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
        ).get(0);
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGetById(id);
        assumeThat("получено ключевое слово", keywords, hasSize(1));
        assertThat("AutotargetingCategories = null",
                keywords.get(0).getAutotargetingCategories().isNil(), equalTo(true));
    }

    @Test
    public void autotargetingCategoriesAndInvalidTypeSOAP() {
        api.protocol(ProtocolType.SOAP);
        Long mobileContentCampaignId = api.userSteps.campaignSteps().addDefaultMobileAppCampaign(CLIENT);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultMobileGroup(mobileContentCampaignId);
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
        ).get(0);
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGetById(id);
        assumeThat("получено ключевое слово", keywords, hasSize(1));
        assertThat("AutotargetingCategories = null",
                keywords.get(0).getAutotargetingCategories().isNil(), equalTo(true));
    }

    @Test
    public void tryToGetAutotargetingCategoriesForKeyword() {
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortKeyword()
                .withAdGroupId(adGroupId)
        ).get(0);

        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGetById(id);
        assumeThat("получено ключевое слово", keywords, hasSize(1));

        assertThat("получены корректные категории автотаргетинга",
                keywords.get(0).getAutotargetingCategories(), nullValue());
    }
}
