package ru.yandex.autotests.directapi.keywords.get;

import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.BidGetItem;
import com.yandex.direct.api.v5.bids.GetResponse;
import com.yandex.direct.api.v5.campaigns.TextCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.TextCampaignSearchStrategyTypeEnum;
import com.yandex.direct.api.v5.general.PriorityEnum;
import com.yandex.direct.api.v5.keywords.KeywordFieldEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.bids.BidExpectedResult;
import ru.yandex.autotests.directapi.model.api5.bids.BidSetItemMap;
import ru.yandex.autotests.directapi.model.api5.bids.SetRequestMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordGetItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 19.03.2015.
 */
@Aqua.Test
@Features(KeywordsFeatures.GET)
@Description("Идентичность значений ставок возвращаемых сервисом Keywords и Bids")
@Tag(TagDictionary.TRUNK)
public class GetKeywordsBidsTest {
    private static LogSteps log = LogSteps.getLogger(GetKeywordsBidsTest.class);
    private static final String client = KeywordsLogins.SINGLE_CLIENT_GET;

    private static Long campaignManualID;
    private static long groupID1;
    private static long keywordID1;
    private static Long autotargetingId1;

    private static Long campaignAutoID;
    private static long groupID2;
    private static long keywordID2;
    private static Long autotargetingId2;

    private static final Long MIN_BID = MoneyCurrency.get(Currency.RUB).getMinPrice().bidLong().longValue();
    private static final Long NEW_BID = MIN_BID * 10L;
    private static final Long NEW_CONTEXT_BID = MIN_BID * 20L;
    private static final PriorityEnum NEW_PRIORITY = PriorityEnum.LOW;

    private static final List<BidFieldEnum> bidFields = Arrays.asList(BidFieldEnum.KEYWORD_ID, BidFieldEnum.BID, BidFieldEnum.CONTEXT_BID, BidFieldEnum.STRATEGY_PRIORITY);
    private static final List<KeywordFieldEnum> keywordFields = Arrays.asList(KeywordFieldEnum.ID, KeywordFieldEnum.BID, KeywordFieldEnum.CONTEXT_BID, KeywordFieldEnum.STRATEGY_PRIORITY);

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepareKeyword() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        log.info("Создадим кампанию со ставками ручных стратегий отличными от ставок по умолчанию");
        campaignManualID = api.userSteps.campaignSteps().addDefaultTextCampaignWithStrategies(
                new TextCampaignSearchStrategyAddMap().withBiddingStrategyType(TextCampaignSearchStrategyTypeEnum.HIGHEST_POSITION),
                new TextCampaignNetworkStrategyAddMap().withBiddingStrategyType(TextCampaignNetworkStrategyTypeEnum.MAXIMUM_COVERAGE),
                client);
        groupID1 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignManualID);
        keywordID1 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID1);
        autotargetingId1 = api.userSteps.keywordsSteps().addAutotargeting(groupID1);

        log.info("Создадим кампанию с приоритетом автостратегии отличной от приоритета по умолчанию");
        campaignAutoID = api.userSteps.campaignSteps().addDefaultTextCampaignWithStrategies(
                new TextCampaignSearchStrategyAddMap().defaultServingOff(),
                new TextCampaignNetworkStrategyAddMap().defaultAverageCpc(Currency.RUB),
                client);
        groupID2 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignAutoID);
        keywordID2 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID2);
        autotargetingId2 = api.userSteps.keywordsSteps().addAutotargeting(groupID2);

        api.userSteps.bidsSteps().shouldGetBidResultOnSet(
                new SetRequestMap()
                        .withBids(
                                new BidSetItemMap()
                                        .withKeywordId(keywordID1)
                                        .withContextBid(NEW_CONTEXT_BID)
                                        .withBid(NEW_BID),
                                new BidSetItemMap()
                                        .withKeywordId(autotargetingId1)
                                        .withContextBid(NEW_CONTEXT_BID)
                                        .withBid(NEW_BID),
                                new BidSetItemMap()
                                        .withKeywordId(keywordID2)
                                        .withStrategyPriority(NEW_PRIORITY),
                                new BidSetItemMap()
                                        .withKeywordId(autotargetingId2)
                                        .withStrategyPriority(NEW_PRIORITY)
                        ),
                BidExpectedResult.successWithKeywordId(),
                BidExpectedResult.successWithKeywordId(),
                BidExpectedResult.successWithKeywordId(),
                BidExpectedResult.successWithKeywordId()
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3621")
    public void keywordBidValuesTest() {
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGetByIdWithFields(keywordFields, keywordID1);
        assumeThat("получили ключевое слово", keywords, hasSize(1));
        GetResponse bidResponse = api.userSteps.bidsSteps().bidsGetByKeywordIdsWithFields(bidFields, keywordID1);
        assumeThat("получили значения ставок", bidResponse.getBids(), hasSize(1));

        KeywordGetItemMap expectedKeyword = makeKeywordItemFromBidItem(bidResponse.getBids().get(0));
        assertThat("идентичные ставки из сервисов Bids и Keywords",
                keywords.get(0),
                beanDifferV5(expectedKeyword.getBean())
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3622")
    public void priorityValuesTest() {
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGetByIdWithFields(keywordFields, keywordID2);
        assumeThat("получили ключевое слово", keywords, hasSize(1));
        GetResponse bidResponse = api.userSteps.bidsSteps().bidsGetByKeywordIdsWithFields(bidFields, keywordID2);
        assumeThat("получили значения ставок", bidResponse.getBids(), hasSize(1));

        KeywordGetItemMap expectedKeyword = makeKeywordItemFromBidItem(bidResponse.getBids().get(0));
        assertThat("идентичные ставки из сервисов Bids и Keywords",
                keywords.get(0),
                beanDifferV5(expectedKeyword.getBean()));
    }

    @Test
    public void autotargetingBidValuesTest() {
        List<KeywordGetItem> autotargetings = api.userSteps.keywordsSteps().keywordsGetByIdWithFields(keywordFields, autotargetingId1);
        assumeThat("получили автотаргетинг", autotargetings, hasSize(1));
        GetResponse bidsResponse = api.userSteps.bidsSteps().bidsGetByKeywordIdsWithFields(bidFields, autotargetingId1);
        assumeThat("получили значения ставок для одного автотаргетинга", bidsResponse.getBids(), hasSize(1));

        KeywordGetItemMap expectedKeyword = makeKeywordItemFromBidItem(bidsResponse.getBids().get(0));
        assertThat("идентичные ставки из сервисов Bids и Keywords",
                autotargetings.get(0),
                beanDifferV5(expectedKeyword.getBean())
        );
    }

    @Test
    public void autotargetingStrategyPriorityValuesTest() {
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGetByIdWithFields(keywordFields, autotargetingId2);
        assumeThat("получили автотаргетинг", keywords, hasSize(1));
        GetResponse bidsResponse = api.userSteps.bidsSteps().bidsGetByKeywordIdsWithFields(bidFields, autotargetingId2);
        assumeThat("получили значения ставок для одного автотаргетинга", bidsResponse.getBids(), hasSize(1));

        KeywordGetItemMap expectedKeyword = makeKeywordItemFromBidItem(bidsResponse.getBids().get(0));
        assertThat("идентичные ставки из сервисов Bids и Keywords",
                keywords.get(0),
                beanDifferV5(expectedKeyword.getBean())
        );
    }

    private KeywordGetItemMap makeKeywordItemFromBidItem(BidGetItem bidItem) {
        return new KeywordGetItemMap()
                .withBid(bidItem.getBid())
                .withContextBid(bidItem.getContextBid())
                .withStrategyPriority(bidItem.getStrategyPriority().getValue())
                .withId(bidItem.getKeywordId());
    }
}
