package ru.yandex.autotests.directapi.keywords.get;


import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import com.yandex.direct.api.v5.keywords.KeywordFieldEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.apache.commons.lang3.StringUtils;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.ReflectionUtils;
import ru.yandex.autotests.direct.utils.converter.EnumToStringValueConverter;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.convert;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 18.03.2015.
 */
@Aqua.Test
@Features(KeywordsFeatures.GET)
@Description("Проверка выдачи при использовании фильтра по FieldNames")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class GetKeywordsFieldNamesTest {
    private static LogSteps log = LogSteps.getLogger(GetKeywordsFieldNamesTest.class);
    private static final String client = KeywordsLogins.SINGLE_CLIENT_GET;

    private static Long campaignID;
    private static long groupID;

    private static long keywordID1;
    private static long keywordID2;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public String criteriaName;

    @Parameterized.Parameter(value = 2)
    public KeywordsSelectionCriteriaMap criteria;

    @Parameterized.Parameter(value = 3)
    public List<KeywordFieldEnum> fieldNames;

    @Parameterized.Parameters(name = "criteria = {1}, test = {0}")
    public static Collection fieldsValue() {
        prepareKeywords();
        Object[][] data = new Object[][]{
                {"Поле StrategyPriority", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(KeywordFieldEnum.STRATEGY_PRIORITY)},
                {"Поле Bid", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(KeywordFieldEnum.BID)},
                {"Поле ContextBid", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(KeywordFieldEnum.CONTEXT_BID)},
                {"Поле StatisticsNetwork", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(KeywordFieldEnum.STATISTICS_NETWORK)},
                {"Обязательное поле", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(KeywordFieldEnum.KEYWORD)},
                {"Опциональное поле заданное", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(KeywordFieldEnum.USER_PARAM_1)},
                {"Опциональное поле не заданное", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(KeywordFieldEnum.USER_PARAM_2)},
                {"Опциональное поле повторяющееся", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(KeywordFieldEnum.USER_PARAM_1, KeywordFieldEnum.USER_PARAM_1)},
                {"Дополнительное составное поле", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(KeywordFieldEnum.PRODUCTIVITY)},
                {"Дополнительное простое поле", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(KeywordFieldEnum.BID)},
                //DIRECT-43127
                {"Дополнительное перечислимое поле", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(KeywordFieldEnum.STATUS)},
                {"Дополнительное поле повторяющеся", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(KeywordFieldEnum.PRODUCTIVITY, KeywordFieldEnum.PRODUCTIVITY)},
                {"Обязательные поля", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID2),
                        Arrays.asList(
                                KeywordFieldEnum.AD_GROUP_ID,
                                KeywordFieldEnum.KEYWORD)},
                {"Обязательные и опциональные поля", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(
                                KeywordFieldEnum.USER_PARAM_1,
                                KeywordFieldEnum.KEYWORD)},
                {"Обязательные, опциональные и дополнительные поля", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID1),
                        Arrays.asList(
                                KeywordFieldEnum.USER_PARAM_1,
                                KeywordFieldEnum.KEYWORD,
                                KeywordFieldEnum.CONTEXT_BID)},
                {"Несколько ID - дополнительное поле", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID2, keywordID1),
                        Arrays.asList(KeywordFieldEnum.PRODUCTIVITY)},
                {"Несколько ID - обязательные, опциональные и дополнительные поля", "keywordId",
                        new KeywordsSelectionCriteriaMap().withIds(keywordID2, keywordID1),
                        Arrays.asList(
                                KeywordFieldEnum.ID,
                                KeywordFieldEnum.AD_GROUP_ID,
                                KeywordFieldEnum.CAMPAIGN_ID,
                                KeywordFieldEnum.KEYWORD,
                                KeywordFieldEnum.USER_PARAM_1,
                                KeywordFieldEnum.STATE,
                                KeywordFieldEnum.CONTEXT_BID,
                                KeywordFieldEnum.PRODUCTIVITY)},
                {"Повторяющиеся ID", "keywordId", new KeywordsSelectionCriteriaMap().withIds(keywordID2, keywordID1, keywordID2),
                        Arrays.asList(
                                KeywordFieldEnum.ID,
                                KeywordFieldEnum.KEYWORD)},

                //по adGroupId
                {"Обязательные, опциональные и дополнительные поля", "adGroupId",
                        new KeywordsSelectionCriteriaMap().withAdGroupIds(groupID),
                        Arrays.asList(
                                KeywordFieldEnum.ID,
                                KeywordFieldEnum.AD_GROUP_ID,
                                KeywordFieldEnum.CAMPAIGN_ID,
                                KeywordFieldEnum.KEYWORD,
                                KeywordFieldEnum.USER_PARAM_1,
                                KeywordFieldEnum.STATE,
                                KeywordFieldEnum.CONTEXT_BID,
                                KeywordFieldEnum.PRODUCTIVITY)},

                //по campaignId
                {"Обязательные, опциональные и дополнительные поля", "campaignId",
                        new KeywordsSelectionCriteriaMap().withCampaignIds(campaignID.longValue()),
                        Arrays.asList(
                                KeywordFieldEnum.ID,
                                KeywordFieldEnum.AD_GROUP_ID,
                                KeywordFieldEnum.CAMPAIGN_ID,
                                KeywordFieldEnum.KEYWORD,
                                KeywordFieldEnum.USER_PARAM_1,
                                KeywordFieldEnum.STATE,
                                KeywordFieldEnum.CONTEXT_BID,
                                KeywordFieldEnum.PRODUCTIVITY)}
        };
        return Arrays.asList(data);
    }

    private static void prepareKeywords() {
        log.info("Подготовим ключевые слова");
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);

        List<Long> ids = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap().shortKeyword().withAdGroupId(groupID).withUserParam1("param1"));
        assumeThat("добавлено одно ключевое слово", ids, hasSize(1));
        keywordID1 = ids.get(0);
        keywordID2 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3627")
    public void filedNamesTest() {
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGet(
                fieldNames, criteria);
        assumeThat("получены ключевые слова", keywords, hasSize(greaterThan(0)));

        log.info("Проверим набор ключевых слов на наличие запрошенных полей");
        for (KeywordGetItem item : keywords) {
            checkFieldNames(item, fieldNames);
        }
    }

    private void checkFieldNames(KeywordGetItem item, List<KeywordFieldEnum> fields) {
        List<String> gotFileds = getNonNullFieldsNames(item);

        //exclude field duplicates
        HashSet<KeywordFieldEnum> fieldsSet = new HashSet<>(fields);
        List<KeywordFieldEnum> nonDuplicatedFields = new ArrayList<>(fieldsSet);

        String[] expectedFields = convert(new ArrayList<>(nonDuplicatedFields), new EnumToStringValueConverter())
                .toArray(new String[nonDuplicatedFields.size()]);

        assertThat("вернулся объект с корректным набором запрошенных полей", gotFileds, containsInAnyOrder(expectedFields));
    }

    private List<String> getNonNullFieldsNames(Object object) {
        if (object == null) {
            throw new IllegalArgumentException("Необходимо передать объект для извлечения имен инициализированных полей");
        }
        log.info("Извлечем имена инициализированных полей объекта");
        List<String> nonNullFileds = new ArrayList<>();
        for (Field field : object.getClass().getDeclaredFields()) {
            if (ReflectionUtils.invokeGetter(object, StringUtils.capitalize(field.getName())) != null) {
                nonNullFileds.add(StringUtils.capitalize(field.getName()));
            }
        }
        return nonNullFileds;
    }

    @AfterClass
    public static void putCampaignToRemove(){
        //workaround для корректроного создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().putCampaignToQueue(client, campaignID);
    }
}
