package ru.yandex.autotests.directapi.keywords.get;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.hamcrest.Matcher;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by chicos on 28.04.15.
 */
@Aqua.Test
@Features(KeywordsFeatures.GET)
@RunWith(Parameterized.class)
@Description("Проверка негативных сценариев выборки ключевых слов по параметрам SelectionCriteria")
@Tag(TagDictionary.TRUNK)
public class GetKeywordsSelectionCriteriaNegativeTest {
    private static LogSteps log = LogSteps.getLogger(GetKeywordsSelectionCriteriaNegativeTest.class);

    private static String login = KeywordsLogins.SINGLE_CLIENT_GET;
    private static String anotherLogin = KeywordsLogins.SINGLE_CLIENT;

    private static long keywordID;
    private static long keywordDeleted;

    private static long keywordOfDeletedGroup;

    private static Long campaignID;
    private static long groupID;
    private static Long campaignDeleted;
    private static long groupDeleted;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public String operator;

    @Parameterized.Parameter(value = 2)
    public KeywordsSelectionCriteriaMap criteriaMap;

    @Parameterized.Parameter(value = 3)
    public Matcher expectedResult;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection dataSet() {
        prepareKeywords();
        Object[][] data = new Object[][]{
                {"Удаленное ключевое слово",
                        login, new KeywordsSelectionCriteriaMap().withIds(keywordDeleted),
                        emptyIterable()},
                {"Ключевое слово удаленной группы по keywordId",
                        login, new KeywordsSelectionCriteriaMap().withIds(keywordOfDeletedGroup),
                        emptyIterable()},
                {"Ключевое слово другого пользователя",
                        anotherLogin, new KeywordsSelectionCriteriaMap().withIds(keywordID),
                        emptyIterable()},
                {"Несуществующее ключевое слово",
                        login, new KeywordsSelectionCriteriaMap().withIds(38497248L),
                        emptyIterable()},
                {"Некорректный Id ключевого слова",
                        login, new KeywordsSelectionCriteriaMap().withIds(0l),
                        emptyIterable()},
                {"Корректный и некорректный Id ключевых слов",
                        login, new KeywordsSelectionCriteriaMap().withIds(keywordID, -123l),
                        hasSize(1)},

                {"Некорректный Id группы",
                        login, new KeywordsSelectionCriteriaMap().withAdGroupIds(0l),
                        emptyIterable()},
                {"Ключевое слово удаленной группы по groupId",
                        login, new KeywordsSelectionCriteriaMap().withAdGroupIds(groupDeleted),
                        emptyIterable()},
                {"Ключевое слово удаленной кампании",
                        login, new KeywordsSelectionCriteriaMap().withCampaignIds(campaignDeleted.longValue()),
                        emptyIterable()}
        };
        return Arrays.asList(data);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3635")
    public void negativeCriteriaTest() {
        List<KeywordGetItem> keywords = api.as(operator).userSteps.keywordsSteps().keywordsGet(criteriaMap);
        assertThat("вернулся корректный набор визиток", keywords, expectedResult);
    }

    @AfterClass
    public static void putCampaignToRemove(){
        //workaround для корректроного создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().putCampaignToQueue(login, campaignID);
    }

    private static void prepareKeywords(){
        log.info("Создадим ключевые слова");
        api.as(login);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(login);
        //workaround для создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().removeCampaignsFromQueue(Arrays.asList(campaignID));

        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        keywordID = api.userSteps.keywordsSteps().addDefaultKeyword(groupID);

        keywordDeleted = api.userSteps.keywordsSteps().addDefaultKeyword(groupID);
        api.userSteps.keywordsSteps().keywordsDelete(keywordDeleted);

        campaignDeleted = api.userSteps.campaignSteps().addDefaultTextCampaign(login);
        groupDeleted = api.userSteps.adGroupsSteps().addDefaultGroup(campaignDeleted);
        keywordOfDeletedGroup = api.userSteps.keywordsSteps().addDefaultKeyword(groupDeleted);
        api.userSteps.campaignSteps().campaignsDelete(campaignDeleted);
    }
}
