package ru.yandex.autotests.directapi.keywords.get;


import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.yandex.direct.api.v5.general.StatusEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import com.yandex.direct.api.v5.keywords.KeywordStateSelectionEnum;
import com.yandex.direct.api.v5.keywords.KeywordStatusSelectionEnum;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.extract;
import static ch.lambdaj.Lambda.on;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by chicos on 27.04.2015
 */
@Aqua.Test
@Features(KeywordsFeatures.GET)
@Description("Проверка комбинаций параметров в SelectionCriteria при запросе ключевых слов и автотаргетингов")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class GetKeywordsSelectionCriteriaTest {
    private static LogSteps log = LogSteps.getLogger(GetKeywordsSelectionCriteriaTest.class);
    private static final String client = KeywordsLogins.SINGLE_CLIENT_GET;

    //кампании
    private static Long campaignID1;
    private static Long campaignID2;

    //группы 1-й кампании
    private static Long groupID11;
    private static Long groupID12;

    //группы 2-й кампании
    private static Long groupID23;

    //ключевые слова 1-й группы
    private static Long keywordID11;
    private static Long keywordID12;
    private static Long autotargetingId13;

    //ключевые слова 2-й группы
    private static Long keywordID23;
    private static Long keywordID24;
    private static Long keywordID25;
    private static Long autotargetingId26;

    //ключевые слова 3-й группы
    private static Long keywordID36;
    private static Long keywordID37;
    private static Long keywordID38;
    private static Long autotargetingId39;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public Long[] campaignIds;

    @Parameterized.Parameter(value = 2)
    public Long[] groupIds;

    @Parameterized.Parameter(value = 3)
    public Long[] keywordIds;

    @Parameterized.Parameter(value = 4)
    public KeywordStateSelectionEnum[] states;

    @Parameterized.Parameter(value = 5)
    public KeywordStatusSelectionEnum[] statuses;

    @Parameterized.Parameter(value = 6)
    public List<Long> expectedKeywords;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection fieldsValue() {
        prepareKeywords();
        log.info("Keywords ready!");
        Object[][] data = new Object[][]{
                {"ID одной кампании",
                        new Long[]{campaignID2}, null, null, null, null,
                        Arrays.asList(keywordID36, keywordID37, keywordID38, autotargetingId39)},
                {"ID двух кампаний",
                        new Long[]{campaignID1, campaignID2}, null, null, null, null,
                        Arrays.asList(keywordID11, keywordID12, keywordID23, keywordID24, keywordID25, keywordID36,
                                keywordID37, keywordID38, autotargetingId13, autotargetingId26, autotargetingId39)},
                {"ID одной группы",
                        null, new Long[]{groupID12}, null, null, null,
                        Arrays.asList(keywordID23, keywordID24, keywordID25, autotargetingId26)},
                {"ID двух групп из разных кампаний",
                        null, new Long[]{groupID11, groupID23}, null, null, null,
                        Arrays.asList(keywordID11, keywordID12, keywordID36, keywordID37, keywordID38,
                                autotargetingId13, autotargetingId39)},
                {"ID ключевого слова",
                        null, null, new Long[]{keywordID37}, null, null,
                        Arrays.asList(keywordID37)},
                {"ID бесфразного таргетинга",
                        null, null, new Long[]{autotargetingId26}, null, null,
                        Arrays.asList(autotargetingId26)},
                {"ID ключевых слов из разных кампаний",
                        null, null, new Long[]{keywordID11, keywordID37}, null, null,
                        Arrays.asList(keywordID11, keywordID37)},
                {"ID ключевых слов и бесфразных таргетингов",
                        null, null, new Long[]{keywordID11, keywordID37, autotargetingId26, autotargetingId39}, null,
                        null,
                        Arrays.asList(keywordID11, keywordID37, autotargetingId26, autotargetingId39)},
                {"ID кампании и группы",
                        new Long[]{campaignID1}, new Long[]{groupID12}, null, null, null,
                        Arrays.asList(keywordID23, keywordID24, keywordID25, autotargetingId26)},
                {"ID кампании и чужой группы",
                        new Long[]{campaignID1}, new Long[]{groupID23}, null, null, null,
                        Arrays.asList()},
                {"ID кампании и ключевого слова",
                        new Long[]{campaignID2}, null, new Long[]{keywordID36}, null, null,
                        Arrays.asList(keywordID36)},
                {"ID кампании и бесфразного таргетинга",
                        new Long[]{campaignID2}, null, new Long[]{autotargetingId39}, null, null,
                        Arrays.asList(autotargetingId39)},
                {"ID кампании и чужого ключевого слова",
                        new Long[]{campaignID2}, null, new Long[]{keywordID23}, null, null,
                        Arrays.asList()},
                {"ID кампании и бесфразного таргетинга из другой кампании",
                        new Long[]{campaignID2}, null, new Long[]{autotargetingId26}, null, null,
                        Arrays.asList()},
                {"ID кампании, группы, ключевого слова и бесфразного таргетинга",
                        new Long[]{campaignID1}, new Long[]{groupID12}, new Long[]{keywordID23, autotargetingId26},
                        null, null,
                        Arrays.asList(keywordID23, autotargetingId26)},
                {"ID кампании, группы и чужого ключевого слова и бесфразного таргетинга",
                        new Long[]{campaignID1}, new Long[]{groupID12},
                        new Long[]{keywordID11, keywordID23, autotargetingId13, autotargetingId26}, null, null,
                        Arrays.asList(keywordID23, autotargetingId26)},
                {"Status и ID кампании",
                        new Long[]{campaignID1}, null, null, null,
                        new KeywordStatusSelectionEnum[]{KeywordStatusSelectionEnum.ACCEPTED},
                        Arrays.asList(keywordID12, keywordID24, autotargetingId13, autotargetingId26)},
                {"Status и ID двух кампаний",
                        new Long[]{campaignID1, campaignID2}, null, null, null,
                        new KeywordStatusSelectionEnum[]{KeywordStatusSelectionEnum.REJECTED},
                        Arrays.asList(keywordID23, keywordID38)},
                {"Status и ID двух групп",
                        null, new Long[]{groupID11, groupID23}, null, null,
                        new KeywordStatusSelectionEnum[]{KeywordStatusSelectionEnum.DRAFT},
                        Arrays.asList(keywordID11, keywordID36)},
                {"Несколько Status'ов  и ID двух групп",
                        null, new Long[]{groupID12, groupID23}, null, null,
                        new KeywordStatusSelectionEnum[]{KeywordStatusSelectionEnum.DRAFT,
                                KeywordStatusSelectionEnum.REJECTED},
                        Arrays.asList(keywordID23, keywordID25, keywordID36, keywordID38)},
                {"State SUSPENDED и ID двух кампании и двух групп",
                        new Long[]{campaignID1, campaignID2}, new Long[]{groupID11, groupID23}, null,
                        new KeywordStateSelectionEnum[]{KeywordStateSelectionEnum.SUSPENDED}, null,
                        Arrays.asList(keywordID12, keywordID37, autotargetingId39)},
                {"State OFF и ID двух кампаний",
                        new Long[]{campaignID1, campaignID2}, null, null,
                        new KeywordStateSelectionEnum[]{KeywordStateSelectionEnum.OFF}, null,
                        Arrays.asList(keywordID11, keywordID23, keywordID36, keywordID38)},
                {"Несколько State'ов и ID двух кампании",
                        new Long[]{campaignID1, campaignID2}, null, null,
                        new KeywordStateSelectionEnum[]{KeywordStateSelectionEnum.SUSPENDED,
                                KeywordStateSelectionEnum.ON}, null,
                        Arrays.asList(keywordID12, keywordID24, keywordID25, keywordID37, autotargetingId13,
                                autotargetingId26, autotargetingId39)}
        };
        return Arrays.asList(data);
    }

    private static void prepareKeywords() {
        log.info("Cоздадим кампанию 1");
        campaignID1 = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        //workaround для создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().removeCampaignsFromQueue(Arrays.asList(campaignID1));

        //создадим группы для кампании 1
        groupID11 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID1);
        groupID12 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID1);
        //создадим ключевые слова группы 1
        keywordID11 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID11);
        keywordID12 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID11); //to suspend
        autotargetingId13 = api.userSteps.keywordsSteps().addAutotargeting(groupID11);
        //создадим ключевые слова группы 2
        keywordID23 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID12); //to reject
        keywordID24 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID12); //to accept
        keywordID25 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID12); //to suspend
        autotargetingId26 = api.userSteps.keywordsSteps().addAutotargeting(groupID12);
        //выставим нужные статусы
        api.userSteps.keywordsSteps().setKeywordStatus(groupID11, keywordID12, StatusEnum.ACCEPTED);
        api.userSteps.keywordsSteps().setKeywordStatus(groupID12, keywordID23, StatusEnum.REJECTED);
        api.userSteps.keywordsSteps().setKeywordStatus(groupID12, keywordID24, StatusEnum.ACCEPTED);
        api.userSteps.keywordsSteps().keywordsSuspend(keywordID12, keywordID25, autotargetingId26);

        log.info("Cоздадим кампанию 2");
        campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        //workaround для создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().removeCampaignsFromQueue(Arrays.asList(campaignID1));

        //создадим группу для кампании 2
        groupID23 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID2);
        //создадим ключевые слова группы 3
        keywordID36 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID23);
        keywordID37 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID23); //to suspend
        keywordID38 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID23); //to reject
        autotargetingId39 = api.userSteps.keywordsSteps().addAutotargeting(groupID23);
        //выставим нужные статусы
        api.userSteps.keywordsSteps().keywordsSuspend(keywordID37, autotargetingId39);
        api.userSteps.keywordsSteps().setKeywordStatus(groupID23, keywordID37, StatusEnum.ACCEPTED);
        api.userSteps.keywordsSteps().setKeywordStatus(groupID23, keywordID38, StatusEnum.REJECTED);
    }

    @Before
    public void resetUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3636")
    public void getKeywordTest() {
        //DIRECT-42283 closed, workaround removed
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGet(new KeywordsSelectionCriteriaMap()
                .withCampaignIds(campaignIds)
                .withAdGroupIds(groupIds)
                .withIds(keywordIds)
                .withStates(states)
                .withStatuses(statuses));
        List<Long> keywordIds = extract(keywords, on(KeywordGetItem.class).getId());

        assertThat("ответ метода совпадает с ожидаемым набором", keywordIds,
                containsInAnyOrder(expectedKeywords.toArray()));
    }

    @AfterClass
    public static void putCampaignToRemove() {
        //workaround для корректроного создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().putCampaignToQueue(client, campaignID1);
        api.userSteps.campaignSteps().putCampaignToQueue(client, campaignID2);
    }
}
