package ru.yandex.autotests.directapi.keywords.get;


import java.util.List;

import com.yandex.direct.api.v5.keywords.KeywordFieldEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import com.yandex.direct.api.v5.keywords.KeywordStateSelectionEnum;
import com.yandex.direct.api.v5.keywords.KeywordStatusSelectionEnum;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.keywords.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordGetItemMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanEquivalentV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 19.05.2015.
 */
@Aqua.Test
@Features(KeywordsFeatures.GET)
@Description("Тесты специфичные для протокола SOAP")
@Tag(TagDictionary.TRUNK)
public class GetKeywordsSoapTest {
    private static final String client = KeywordsLogins.SINGLE_CLIENT_GET;

    private static long keywordId;
    private static long groupId;
    private static Long campaignId;
    private static final String PATTERN_API = "yyyy-MM-dd'T'HH:mm:ss'Z'";
    private static String modifiedSince = DateTime.now().minusHours(12).toString(PATTERN_API);

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client).protocol(ProtocolType.SOAP);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepareKeyword() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        groupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        keywordId = api.userSteps.keywordsSteps().addDefaultKeyword(groupId);
    }

    @Test
    @Description("Запрос ключевого слова по протоколу SOAP")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3638")
    public void getKeywordBySoapTest() {
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGetById(keywordId);
        assumeThat("получено ключевое слово", keywords, hasSize(1));

        KeywordGetItemMap expectedKeyword = new KeywordGetItemMap().withAdGroupId(groupId).withId(keywordId);
        assertThat("корректное ключевое слово", keywords.get(0),
                beanEquivalentV5(expectedKeyword.getBean()));
    }

    @Test
    @Description("Запрос ключевого слова по протоколу SOAP с ModifiedSince")
    public void getKeywordWithModifiedSinceBySoapTest() {
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGet(
                new GetRequestMap()
                        .withSelectionCriteria(new KeywordsSelectionCriteriaMap()
                                .withIds(keywordId)
                                .withModifiedSince(modifiedSince))
                        .withFieldNames(KeywordFieldEnum.ID)
        );
        assumeThat("получено ключевое слово", keywords, hasSize(1));

        KeywordGetItemMap expectedKeyword = new KeywordGetItemMap().withId(keywordId);
        assertThat("корректное ключевое слово", keywords.get(0),
                beanEquivalentV5(expectedKeyword.getBean()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3637")
    public void noSelectionCriteriaInRequestTest() {
        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.GET,
                new GetRequestMap().withFieldNames(KeywordFieldEnum.ID),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        StringUtils.capitalize(GetRequestMap.SELECTION_CRITERIA)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3639")
    public void noFieldNamesTest() {
        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new KeywordsSelectionCriteriaMap().withIds(keywordId)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        StringUtils.capitalize(GetRequestMap.FIELD_NAMES)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3640")
    public void invalidStatusEnumTest() {
        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new KeywordsSelectionCriteriaMap()
                                .withIds(keywordId)
                                .withStates(KeywordStateSelectionEnum.ON)
                                .withStatuses("INVALID_VALUE"))
                        .withAllFieldNames(),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.ELEMENT_OF_ARRAY_CONTAINS_INCORRECT_ENUMERATE_VALUE,
                        path(GetRequestMap.SELECTION_CRITERIA, KeywordsSelectionCriteriaMap.STATUSES),
                        "DRAFT, ACCEPTED, REJECTED"));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3641")
    public void invalidStateEnumTest() {
        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new KeywordsSelectionCriteriaMap()
                                .withIds(keywordId)
                                .withStatuses(KeywordStatusSelectionEnum.DRAFT)
                                .withStates("INVALID_VALUE"))
                        .withAllFieldNames(),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.ELEMENT_OF_ARRAY_CONTAINS_INCORRECT_ENUMERATE_VALUE,
                        path(GetRequestMap.SELECTION_CRITERIA, KeywordsSelectionCriteriaMap.STATES),
                        "OFF, ON, SUSPENDED"));
    }
}
