package ru.yandex.autotests.directapi.keywords.get;

import com.yandex.direct.api.v5.keywords.KeywordStatusSelectionEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.keywords.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 15.05.15
 */
@Aqua.Test
@Features(KeywordsFeatures.GET)
@Stories(ApiStories.UNITS)
@Description("Списание баллов при запросе ключевых слов")
public class GetKeywordsUnitsTest {
    private static LogSteps log = LogSteps.getLogger(GetKeywordsUnitsTest.class);
    private static final String singleClient = KeywordsLogins.UNITS_GET_CLIENT;

    private static final String agency = KeywordsLogins.AGENCY_UNITS_GET;
    private static final String subclient = KeywordsLogins.SUBCLIENT_UNITS_GET;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int GET_COST = 15;
    private static final int ITEM_COST_ON_GET = 0;
    private static final int GENERAL_FAULT_COST = 50;

    private static Long campaignID;
    private static long groupID;

    private static long keywordID1;
    private static long keywordID2;

    @BeforeClass
    public static void createGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(singleClient);
        campaignID = api.as(singleClient).userSteps.campaignSteps().addDefaultTextCampaign(singleClient);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);

        keywordID1 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID);
        keywordID2 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID);
    }

    @Before
    @Step("Сбросим использованные баллы")
    public void prepareClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(singleClient);
    }

    @Test
    @Description("Получение одиночного ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3657")
    public void getSingleKeywordTest() {
        api.as(singleClient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.keywordsSteps().keywordsGetById(keywordID1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assertThat("стоимость получения одного ключевого слова", unitsBefore - unitsAfter, equalTo(GET_COST + ITEM_COST_ON_GET));
    }

    @Test
    @Description("Получение нескольких ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3658")
    public void getSeveralKeywordsTest() {
        api.as(singleClient);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.keywordsSteps().keywordsGetById(keywordID1, keywordID2);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость получения двух ключевых слов", unitsBefore - unitsAfter, equalTo(GET_COST + ITEM_COST_ON_GET * 2));
    }

    @Test
    @Description("Получение ключевого слова по AdGroupId")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3659")
    public void getKeywordByGroupIDTest() {
        api.as(singleClient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.keywordsSteps().keywordsGetByAdGroupId(groupID);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assertThat("стоимость получения одного ключевого слова", unitsBefore - unitsAfter, equalTo(GET_COST + ITEM_COST_ON_GET * 2));
    }

    @Test
    @Description("Получение ключевого слова по CampaignId и Status")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3656")
    public void getKeywordByCampaignIDAndStatusTest() {
        api.as(singleClient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.keywordsSteps().keywordsGet(new GetRequestMap()
                .withSelectionCriteria(new KeywordsSelectionCriteriaMap()
                        .withCampaignIds(campaignID.longValue())
                        .withStatuses(KeywordStatusSelectionEnum.DRAFT))
                .withAllFieldNames());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assertThat("стоимость получения одного ключевого слова", unitsBefore - unitsAfter, equalTo(GET_COST + ITEM_COST_ON_GET * 2));
    }

    @Test
    @Description("Запрос несуществующего ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3660")
    public void getNonExistingKeywordTest() {
        api.as(singleClient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.keywordsSteps().keywordsGetById(1234l);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость запроса несуществующего ключевого слова", unitsBefore - unitsAfter, equalTo(GET_COST));
    }

    @Test
    @Description("Общая ошибка при получении ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3661")
    public void generalErrorOnGetKeywordTest() {
        api.as(singleClient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new KeywordsSelectionCriteriaMap()
                                .withIds(keywordID1)),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        org.apache.commons.lang3.StringUtils.capitalize(GetRequestMap.FIELD_NAMES)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость общей ошибки при получении ключевого слова", unitsBefore - unitsAfter, equalTo(GENERAL_FAULT_COST));
    }

    @Test
    @Description("Недостаточное колличество баллов при получении ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3662")
    public void notEnoughUnitsOnGetKeywordTest() {
        api.as(singleClient);
        log.info("Обнулим колличество доступных пользователю баллов");
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(singleClient, unitsBefore);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assumeThat("доступные баллы пользователя", unitsBefore, equalTo(0));

        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.GET,
                new GetRequestMap()
                        .withSelectionCriteria(new KeywordsSelectionCriteriaMap()
                                .withIds(keywordID1))
                        .withAllFieldNames(),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        assertThat("доступные баллы пользователя после ошибки", unitsAfter, equalTo(0));
    }

    @Test
    @Description("Получение агентством нескольких ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3655")
    public void addKeywordsUnitsCostByAgencyTest() {
        api.as(agency);
        log.info("Подготовим ключевые слова");
        Long agencyCampaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(subclient);
        long groupID = api.userSteps.adGroupsSteps().addDefaultGroup(agencyCampaignID, subclient);
        long agencyKeyword1 = api.userSteps.keywordsSteps().addDefaultKeyword(subclient, groupID);
        long agencyKeyword2 = api.userSteps.keywordsSteps().addDefaultKeyword(subclient, groupID);

        int agencyUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int scUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        api.userSteps.keywordsSteps().keywordsGet(subclient,
                new GetRequestMap()
                        .withSelectionCriteria(new KeywordsSelectionCriteriaMap()
                                .withIds(agencyKeyword1, agencyKeyword2))
                        .withAllFieldNames());
        int scUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        int agencyUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);

        assertThat("списание баллов у агентства", agencyUnitsBefore - agencyUnitsAfter, equalTo(0));
        assertThat("списание баллов у субклиента", scUnitsBefore - scUnitsAfter, equalTo(GET_COST + 2 * ITEM_COST_ON_GET));
    }
}
