package ru.yandex.autotests.directapi.keywords.get;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.IntStream;

import com.yandex.direct.api.v5.keywords.KeywordFieldEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import one.util.streamex.StreamEx;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.UsersPreselectedData;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.general.LimitOffsetMap;
import ru.yandex.autotests.directapi.model.api5.keywords.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by hmepas on 08.12.16
 */

@Aqua.Test
@Features(KeywordsFeatures.GET)
@Stories(ApiStories.UNITS)
@RunWith(Parameterized.class)
@Description("Списание баллов при запросе ключевых слов с данными из БК или ADVQ")
public class GetMoreThan4000KeywordsUnitsTest {

    private static LogSteps log = LogSteps.getLogger(GetMoreThan4000KeywordsUnitsTest.class);

    private static final int GET_COST = 15;
    private static final int ITEM_COST_ON_GET_PER_2000_WITH_STAT = 3;
    private static final int ITEM_COST_ON_GET_PER_2000_WITHOUT_STAT = 1;

    private static final String LOGIN = UsersPreselectedData.MORE_THAT_4000_KEYWORDS_LOGIN;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);


    @Parameterized.Parameter
    public String message;

    @Parameterized.Parameter(value = 1)
    public KeywordFieldEnum[] fieldNames;

    @Parameterized.Parameter(value = 2)
    public int costPerExtra2000;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection fieldsValue() {
        Object[][] data = new Object[][]{
                {"Получение каждых 2000 фраз сверх первых двух тысяч, без статистики стоит в баллах: " + ITEM_COST_ON_GET_PER_2000_WITHOUT_STAT,
                        new KeywordFieldEnum[] {
                                KeywordFieldEnum.ID,
                                KeywordFieldEnum.STATUS,
                                KeywordFieldEnum.STATE,
                                KeywordFieldEnum.KEYWORD,
                                KeywordFieldEnum.AD_GROUP_ID,
                                KeywordFieldEnum.BID,
                                KeywordFieldEnum.CONTEXT_BID,
                                KeywordFieldEnum.SERVING_STATUS,
                                KeywordFieldEnum.CAMPAIGN_ID,
                                KeywordFieldEnum.STRATEGY_PRIORITY,
                                KeywordFieldEnum.USER_PARAM_1,
                                KeywordFieldEnum.USER_PARAM_2},
                        ITEM_COST_ON_GET_PER_2000_WITHOUT_STAT},

                {"Получение каждых 2000 фраз сверх первых двух тысяч, с продуктивностю стоит в баллах: " + ITEM_COST_ON_GET_PER_2000_WITH_STAT,
                        new KeywordFieldEnum[] {
                                KeywordFieldEnum.PRODUCTIVITY
                        },
                        ITEM_COST_ON_GET_PER_2000_WITH_STAT},
                {"Получение каждых 2000 фраз сверх первых двух тысяч, со статистикой на поиске стоит в баллах: " + ITEM_COST_ON_GET_PER_2000_WITH_STAT,
                        new KeywordFieldEnum[] {
                                KeywordFieldEnum.STATISTICS_SEARCH
                        },
                        ITEM_COST_ON_GET_PER_2000_WITH_STAT},
                {"Получение каждых 2000 фраз сверх первых двух тысяч, со статистикой на сети стоит в баллах: " + ITEM_COST_ON_GET_PER_2000_WITH_STAT,
                        new KeywordFieldEnum[] {
                                KeywordFieldEnum.STATISTICS_NETWORK
                        },
                        ITEM_COST_ON_GET_PER_2000_WITH_STAT},
        };
        return Arrays.asList(data);
    }

    private static long campaignId;

    @BeforeClass
    public static void createKeywords() {
        api.as(Logins.SUPER_LOGIN, LOGIN);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        AdGroupAddItemMap adGroupAddItemMap = new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId);

        AtomicInteger recordCounter = new AtomicInteger(0);
        StreamEx
                .of(IntStream.range(1, 4400).mapToObj(value -> "Keyword-" + value))
                .groupRuns((prev, next) -> recordCounter.incrementAndGet() % 200 != 0)
                .forEach(keywords -> {
                    long adGroupId = api.userSteps.adGroupsSteps().addGroup(adGroupAddItemMap.withName("Group-" + recordCounter.get()), LOGIN);
                    api.userSteps.keywordsSteps().addKeywords(LOGIN, adGroupId, keywords);
                });
    }

    @Test
    @Description("Получение каждых 2000 фраз сверх первой, стоимость в баллах в зависимости от полей")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3658")
    public void getMoreThat2000KeywordsNoStatTest() {
        api.as(Logins.SUPER_LOGIN, LOGIN);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        final List<KeywordGetItem> keywordGetItems = api.userSteps.keywordsSteps().keywordsGet(
                new GetRequestMap().withSelectionCriteria(
                        new KeywordsSelectionCriteriaMap().withCampaignIds(campaignId)
                ).withFieldNames(
                        fieldNames
                ).withPage(new LimitOffsetMap().withLimit(4000L))
        );
        assumeThat("вернулось более 4000 фраз", keywordGetItems, hasSize(greaterThanOrEqualTo(4000)));

        int size = keywordGetItems.size();
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);

        assertThat(message, unitsBefore - unitsAfter, equalTo(GET_COST + costPerExtra2000 * ( (int) size/2000) ));
    }
}
