package ru.yandex.autotests.directapi.keywords.resume;


import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.CampaignFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 22.05.2015.
 */
@Aqua.Test
@Features(KeywordsFeatures.RESUME)
@Description("Проверка у группы времени последнего изменения и статуса синхронизации с БК при возобновлении ключевого слова")
@Tag(TagDictionary.TRUNK)
public class ResumeKeywordsAdGroupLastChangeAndBsSyncedTest {
    private static LogSteps log = LogSteps.getLogger(ResumeKeywordsAdGroupLastChangeAndBsSyncedTest.class);
    private static final String client = KeywordsLogins.SINGLE_CLIENT_RESUME;

    private static Long campaignID;
    private static Long adID;
    private static Long groupID;
    private static Long keywordID;
    private static DateTime lastChange = DateTime.now().minusDays(5);
    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @BeforeClass
    public static void prepareGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        adID = api.userSteps.adsSteps().addDefaultTextAd(groupID);
        api.userSteps.keywordsSteps().addDefaultKeyword(groupID);
        keywordID = api.userSteps.keywordsSteps().addDefaultKeyword(groupID);
        api.userSteps.keywordsSteps().keywordsSuspend(keywordID);

        log.info("Промодерируем фейково");
        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignID);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(groupID);
        api.userSteps.bannersFakeSteps().makeBannersModerated(adID);

        log.info("Выставим статусы синхронизации с БК фейково");
        api.userSteps.campaignFakeSteps().setBSSynced(campaignID, true);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(groupID, Status.YES);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adID, Status.YES);

        log.info("Сдвинем время последнего изменения на несколько дней");
        api.userSteps.campaignFakeSteps().setLastChange(campaignID, lastChange.toString());
        api.userSteps.groupFakeSteps().setLastChange(groupID, lastChange.toString());
        api.userSteps.bannersFakeSteps().setLastChange(adID, lastChange.toString());

        log.info("Возобновим ключевое слово");
        api.userSteps.keywordsSteps().keywordsResume(keywordID);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3677")
    public void campaignStatusBsSyncedTest() {
        CampaignFakeInfo campaignInfo = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID);

        assertThat("статус синхронизации с БК у кампании после возобновления ключевого слова",
                campaignInfo.getStatusBsSynced(), equalTo(Status.YES));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3678")
    public void bannerStatusBsSyncedTest() {
        BannerFakeInfo bannerInfo = api.userSteps.bannersFakeSteps().getBannerParams(adID);

        assertThat("статус синхронизации с БК у баннера после возобновления ключевого слова",
                bannerInfo.getStatusBsSynced(), equalTo(Status.YES));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3679")
    public void groupStatusBsSyncedTest() {
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(groupID);

        assertThat("статус синхронизации с БК у группы после возобновления ключевого слова",
                groupInfo.getStatusBsSynced(), equalTo(Status.NO));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3680")
    public void campaignLastChangeTest() {
        CampaignFakeInfo campaignInfo = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID);
        assumeThat("время последнего изменения кампании", campaignInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(campaignInfo.getLastChange());

        assertThat("время последнего изменения кампании после возобновления ключевого слов", date, equalTo(lastChange.toLocalDate()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3681")
    public void bannerLastChangeTest() {
        BannerFakeInfo bannerInfo = api.userSteps.bannersFakeSteps().getBannerParams(adID);
        assumeThat("время последнего изменения кампании", bannerInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(bannerInfo.getLastChange());

        assertThat("время последнего изменения баннера после возобновления ключевого слова", date, equalTo(lastChange.toLocalDate()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3682")
    public void groupLastChangeTest() {
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(groupID);
        assumeThat("время последнего изменения группы", groupInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(groupInfo.getLastChange());

        assertThat("время последнего изменения группы после возобновления ключевого слова", date, equalTo(LocalDate.now()));
    }
}
