package ru.yandex.autotests.directapi.keywords.resume;

import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.general.StateEnum;
import com.yandex.direct.api.v5.general.StatusEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordGetItemMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.extract;
import static ch.lambdaj.Lambda.on;
import static ch.lambdaj.Lambda.sort;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.everyItem;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by semkagtn on 12.03.15.
 * https://st.yandex-team.ru/TESTIRT-4391
 */
@Aqua.Test
@Features(KeywordsFeatures.RESUME)
@Issue("https://st.yandex-team.ru/DIRECT-39271")
@Description("Проверка метода Keyword.Resume")
@Tag(TagDictionary.TRUNK)
public class ResumeTest {

    private static final String LOGIN = KeywordsLogins.SINGLE_CLIENT_RESUME;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static long adGroupID;
    private long firstKeywordID;
    private long secondKeywordID;
    private Long autotargetingId;

    @BeforeClass
    public static void createAdGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createKeywords() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        firstKeywordID = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
        secondKeywordID = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
        autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupID);

        api.userSteps.keywordsSteps().setKeywordStatus(adGroupID, firstKeywordID, StatusEnum.ACCEPTED);
        api.userSteps.keywordsSteps().setKeywordStatus(adGroupID, secondKeywordID, StatusEnum.ACCEPTED);

        api.userSteps.keywordsSteps().keywordsSuspend(firstKeywordID, secondKeywordID,autotargetingId);
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGet(new KeywordsSelectionCriteriaMap()
                .withIds(firstKeywordID, secondKeywordID,autotargetingId));
        List<StateEnum> states = extract(keywords, on(KeywordGetItem.class).getState());
        assumeThat("оба ключевых слова и автотаргетинг отключены", states, everyItem(equalTo(StateEnum.SUSPENDED)));
    }

    @Test
    @Description("Включить одно ключевое слово")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3693")
    public void resumeOneKeyword() {
        api.userSteps.keywordsSteps().keywordsResume(firstKeywordID);
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGet(new KeywordsSelectionCriteriaMap()
                .withIds(firstKeywordID, secondKeywordID));
        keywords = sort(keywords, on(KeywordGetItem.class).getId());

        assumeThat("было включено верное ключевое слово", keywords, beanEquivalent(Arrays.asList(
                (KeywordGetItem) new KeywordGetItemMap()
                        .withId(firstKeywordID)
                        .withState(StateEnum.ON).getBean(),
                (KeywordGetItem) new KeywordGetItemMap()
                        .withId(secondKeywordID)
                        .withState(StateEnum.SUSPENDED).getBean())));
    }

    @Test
    @Description("Включить два ключевых слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3694")
    public void resumeTwoKeywords() {
        api.userSteps.keywordsSteps().keywordsResume(firstKeywordID, secondKeywordID);
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGet(new KeywordsSelectionCriteriaMap()
                .withIds(firstKeywordID, secondKeywordID));
        List<StateEnum> states = extract(keywords, on(KeywordGetItem.class).getState());
        assertThat("оба ключевых слова включены", states, everyItem(equalTo(StateEnum.ON)));
    }

    @Test
    @Description("Включить автотаргетинг")
    public void resumeAutotargeting() {
        api.userSteps.keywordsSteps().keywordsResume(autotargetingId);
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGet(
                new KeywordsSelectionCriteriaMap().withIds(autotargetingId)
        );
        assertThat("автотаргетинг включен", keywords.get(0).getState(), equalTo(StateEnum.ON));
    }

    @Test
    @Description("Включить два ключевых слова и автотаргетинг")
    public void resumeTwoKeywordsAndAutotargeting() {
        api.userSteps.keywordsSteps().keywordsResume(firstKeywordID, secondKeywordID,autotargetingId);
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGet(
                new KeywordsSelectionCriteriaMap().withIds(firstKeywordID, secondKeywordID,autotargetingId)
        );
        assumeThat("получили три объекта", keywords.size(), equalTo(3));
        List<StateEnum> states = extract(keywords, on(KeywordGetItem.class).getState());
        assertThat("оба ключевых слова и автотаргетинг включены", states, everyItem(equalTo(StateEnum.ON)));
    }
}
