package ru.yandex.autotests.directapi.keywords.resume;

import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.general.StateEnum;
import com.yandex.direct.api.v5.general.StatusEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordGetItemMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordUpdateMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.keywords.ResumeRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by semkagtn on 12.03.15.
 * https://st.yandex-team.ru/TESTIRT-4391
 */
@Aqua.Test
@Features(KeywordsFeatures.RESUME)
@Stories(ApiStories.UNITS)
@Issue("https://st.yandex-team.ru/DIRECT-39271")
@Description("Проверка списания баллов при включении ключевых слов")
public class ResumeUnitsTest {

    private static final String LOGIN = KeywordsLogins.UNITS_RESUME_CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int CALL_COST = 15;
    private static final int KEYWORD_COST = 0;
    private static final int ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR_COST = 50;

    private static long adGroupID;

    @BeforeClass
    public static void createAdGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
    }

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    @Description("Включение одного ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3696")
    public void resumeOneKeyword() {
        long keywordID = createAndSuspendKeyword(adGroupID);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().keywordsResume(keywordID);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + KEYWORD_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Включение двух ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3697")
    public void resumeTwoKeywords() {
        long firstKeywordID = createAndSuspendKeyword(adGroupID);
        long secondKeywordID = createAndSuspendKeyword(adGroupID);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().keywordsResume(firstKeywordID, secondKeywordID);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + KEYWORD_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Включение одного невалидного ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3698")
    public void resumeOneInvalidKeyword() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.RESUME,
                new ResumeRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(0L)),
                ExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetailsJava.ID_MUST_BE_POSITIVE_INTEGER,
                                path(KeywordUpdateMap.ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Включение двух невалидных ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3699")
    public void resumeTwoInvalidKeywords() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.RESUME,
                new ResumeRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(0l, -1l)),
                ExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetailsJava.ID_MUST_BE_POSITIVE_INTEGER,
                                path(KeywordUpdateMap.ID))),
                ExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetailsJava.ID_MUST_BE_POSITIVE_INTEGER,
                                path(KeywordUpdateMap.ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Включение одного валидного и одного невалидного ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3700")
    public void resumeOneValidAndOneInvalidKeywords() {
        long keywordID = createAndSuspendKeyword(adGroupID);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.RESUME,
                new ResumeRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(keywordID, 0L)),
                ExpectedResult.success(keywordID),
                ExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetailsJava.ID_MUST_BE_POSITIVE_INTEGER,
                                path(KeywordUpdateMap.ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST + KEYWORD_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Невалидный запрос на включение ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3695")
    public void invalidRequest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.RESUME,
                new ResumeRequestMap()
                        .withSelectionCriteria(null),
                JavaOrPerlApi5Error
                        .java(
                                new Api5Error(8000,
                                        Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                                        path(ResumeRequestMap.SELECTION_CRITERIA))
                        )
                        .perl(
                                new Api5Error(8000,
                                        Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                                        "params",
                                        path(ResumeRequestMap.SELECTION_CRITERIA))
                        ));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = COMMON_REQUEST_ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Включение ключевого слова, когда у клиента нету баллов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3701")
    public void resumeKeywordWhenClientHasZeroUnits() {
        long keywordID = createAndSuspendKeyword(adGroupID);

        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("у клиента нету баллов", units, equalTo(0));

        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.RESUME,
                new ResumeRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(keywordID)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
    }

    private static long createAndSuspendKeyword(long adGroupID) {
        long keywordID = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
        api.userSteps.keywordsSteps().setKeywordStatus(adGroupID, keywordID, StatusEnum.ACCEPTED);
        api.userSteps.keywordsSteps().keywordsSuspend(keywordID);
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGet(new KeywordsSelectionCriteriaMap()
                .withIds(keywordID));
        assumeThat("ключевое слово было остановлено", keywords, beanEquivalent(Arrays.asList(
                (KeywordGetItem) new KeywordGetItemMap()
                        .withState(StateEnum.SUSPENDED).getBean())));
        return keywordID;
    }
}
