package ru.yandex.autotests.directapi.keywords.suspend;

import com.yandex.direct.api.v5.general.StateEnum;
import com.yandex.direct.api.v5.general.StatusEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.keywords.SuspendRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by gerdler on 20.04.17.
 * https://st.yandex-team.ru/DIRECT-64666
 */
@Aqua.Test
@Features(KeywordsFeatures.SUSPEND)
@Issue("https://st.yandex-team.ru/DIRECT-64386")
@Description("Остановка автотаргетинга")
public class SuspendAutotargetingTest {

    private static final String CLIENT = KeywordsLogins.SINGLE_CLIENT_SUSPEND;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long campaignId;
    private Long adGroupId;
    private Long autotargetingId;

    @BeforeClass
    public static void createCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createAutotargeting() {
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);

        assumeThat("автотаргетинг находится в статусе On",
                api.userSteps.keywordsSteps().getKeywordState(autotargetingId),
                equalTo(StateEnum.ON)
        );
    }

    @Test
    @Description("Приостановка автотаргетинга без ключевых слов в группе")
    public void suspendAutotargetingWithoutKeywordsInAdGroupTest() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.SUSPEND,
                new SuspendRequestMap().withSelectionCriteria(
                        new IdsCriteriaMap().withIds(autotargetingId)
                ),
                ExpectedResult.success()
        );
    }

    @Test
    @Description("Приостановка автотаргетинга при наличии в группе неактивного ключевого слова")
    public void suspendAutotargetingWithSuspendedKeywordInAdGroupTest() {
        Long suspendedKeywordId = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
        api.userSteps.keywordsSteps().setKeywordStatus(adGroupId, suspendedKeywordId, StatusEnum.ACCEPTED);
        api.userSteps.keywordsSteps().keywordsSuspend(suspendedKeywordId);
        assumeThat("ключевое слово находится в статусе Suspended",
                api.userSteps.keywordsSteps().getKeywordState(suspendedKeywordId),
                equalTo(StateEnum.SUSPENDED)
        );

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.SUSPEND,
                new SuspendRequestMap().withSelectionCriteria(
                        new IdsCriteriaMap().withIds(autotargetingId)
                ),
                ExpectedResult.success()
        );
    }

    @Test
    @Description("Приостановка автотаргетинга при наличии в группе активного ключевого слова")
    public void suspendAutotargetingWithActiveKeywordInAdGroupTest() {
        Long activeKeywordId = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
        api.userSteps.keywordsSteps().setKeywordStatus(adGroupId, activeKeywordId, StatusEnum.ACCEPTED);
        assumeThat("ключевое слово находится в статусе On",
                api.userSteps.keywordsSteps().getKeywordState(activeKeywordId),
                equalTo(StateEnum.ON)
        );

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.SUSPEND,
                new SuspendRequestMap().withSelectionCriteria(
                        new IdsCriteriaMap().withIds(autotargetingId)
                ),
                ExpectedResult.success()
        );
    }

    @Test
    @Description("Приостановка автотаргетинга вместе с ключевым словом")
    public void suspendAutotargetingAndKeywordTest() {
        Long activeKeywordId = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
        api.userSteps.keywordsSteps().setKeywordStatus(adGroupId, activeKeywordId, StatusEnum.ACCEPTED);
        assumeThat("ключевое слово находится в статусе On",
                api.userSteps.keywordsSteps().getKeywordState(activeKeywordId),
                equalTo(StateEnum.ON)
        );

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.SUSPEND,
                new SuspendRequestMap().withSelectionCriteria(
                        new IdsCriteriaMap().withIds(activeKeywordId, autotargetingId)
                ),
                ExpectedResult.success(),
                ExpectedResult.success()
        );
    }
}
