package ru.yandex.autotests.directapi.keywords.suspend;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.general.StateEnum;
import com.yandex.direct.api.v5.general.StatusEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5JsonError;
import ru.yandex.autotests.directapi.apiclient.errors.Api5JsonErrorMatcher;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.SuspendRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 24.03.15.
 * https://st.yandex-team.ru/TESTIRT-4390
 */
@Aqua.Test
@Features(KeywordsFeatures.SUSPEND)
@Issue("https://st.yandex-team.ru/DIRECT-39271")
@Description("Приостановка ключевых слов и автотаргетингов от имени разных ролей")

@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class SuspendRolesTest {

    private static final String CLIENT = KeywordsLogins.SINGLE_CLIENT;
    private static final String CLIENT_REP = KeywordsLogins.SINGLE_CLIENT_REP;
    private static final String AGENCY = KeywordsLogins.AGENCY;
    private static final String AGENCY_REP = KeywordsLogins.AGENCY_REP;
    private static final String SUBCLIENT = KeywordsLogins.SUBCLIENT;
    private static final String MANAGER = Logins.MANAGER_DEFAULT;
    private static final String SERV_CLIENT = KeywordsLogins.SERV_CLIENT;
    private static final String SUPER = Logins.SUPER_LOGIN;
    private static final String SUPER_READER = Logins.SUPER_READER;
    private static final String PLACER = Logins.PLACER;
    private static final String SUPPORT = Logins.SUPPORT;
    private static final String MEDIA = Logins.MEDIA;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String loginToCreateCampaign;

    @Parameterized.Parameter(2)
    public String apiUserLogin;

    @Parameterized.Parameter(3)
    public String login;

    @Parameterized.Parameter(4)
    public ExpectedResult expectedResult;

    @Parameterized.Parameter(5)
    public Api5Error expectedError;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT_REP);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AGENCY);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AGENCY_REP);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(SUBCLIENT);

        return Arrays.asList(new Object[][]{
                {"Выключение от имени менеджера", MANAGER, MANAGER, SERV_CLIENT,
                        ExpectedResult.success(), null},

                {"Выключение от имени вешальщика", SUPER, PLACER, CLIENT,
                        ExpectedResult.errors(new Notification(54, Api5ErrorDetails.NO_RIGHTS_TO_WRITE)),
                        new Api5Error(54)},

                {"Выключение от имени саппорта", SUPER, SUPPORT, CLIENT,
                        ExpectedResult.success(), null},

                {"Выключение от имени супера", SUPER, SUPER, CLIENT,
                        ExpectedResult.success(), null},

                {"Выключение от имени медиапланера", SUPER, MEDIA, CLIENT,
                        ExpectedResult.errors(new Notification(54, Api5ErrorDetails.NO_RIGHTS_TO_WRITE)),
                        new Api5Error(54)},

                {"Выключение от имени суперридера", SUPER, SUPER_READER, CLIENT,
                        ExpectedResult.errors(new Notification(54, Api5ErrorDetails.NO_RIGHTS_TO_WRITE)),
                        new Api5Error(54)},

                {"Выключение от имени клиента для самого себя", SUPER, CLIENT, CLIENT,
                        ExpectedResult.success(), null},

                {"Выключение от имени представителя клиента для клиента", SUPER, CLIENT_REP, CLIENT,
                        ExpectedResult.success(), null},

                {"Выключение от имени агенства для субклиента", AGENCY, AGENCY, SUBCLIENT,
                        ExpectedResult.success(), null},

                {"Выключение от имени представителя агенства для субклиента", AGENCY_REP, AGENCY_REP, SUBCLIENT,
                        ExpectedResult.success(), null},

                {"Выключение от имени субклиента для самого себя", AGENCY, SUBCLIENT, SUBCLIENT,
                        ExpectedResult.errors(new Notification(54, Api5ErrorDetails.NO_RIGHTS_TO_WRITE)),
                        new Api5Error(54)}
        });
    }

    private Long adGroupId;

    @Before
    @Step("Подготовка данных для теста")
    public void addUnitsToUser() {
        api.as(loginToCreateCampaign);
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(login);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID, login);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3725")
    public void suspendKeywordUsingLogin() {
        Long keywordID = api.userSteps.keywordsSteps().addDefaultKeywords(login, adGroupId, 2).get(0);
        api.userSteps.keywordsSteps().setKeywordStatus(adGroupId, keywordID, StatusEnum.ACCEPTED);
        assumeThat("ключевое слово включено",
                api.userSteps.keywordsSteps().getKeywordState(login, keywordID),
                equalTo(StateEnum.ON)
        );
        api.as(apiUserLogin);
        try {
            api.userSteps.keywordsSteps().shouldGetResultOn(
                    Action.SUSPEND,
                    login,
                    new SuspendRequestMap()
                            .withSelectionCriteria(new IdsCriteriaMap()
                                    .withIds(keywordID)),
                    expectedResult);
        } catch (Api5JsonError jsonError) {
            // Java версия сервиса возвращает 54-ю ошибку на весь запрос см. DIRECT-77051
            if (expectedError == null) {
                throw new AssertionError("неожиданная ошибка", jsonError);
            } else {
                assertThat("совпал текст ошибки (JSON)",
                        jsonError, Api5JsonErrorMatcher.equalToIgnoreLocale(expectedError.toJsonError()));
            }
        }
    }

    @Test
    public void suspendAutotargetingUsingLogin() {
        Long autotargetingId =
                api.as(loginToCreateCampaign).userSteps.keywordsSteps().addAutotargeting(login, adGroupId);
        assumeThat("автотаргетинг включен",
                api.userSteps.keywordsSteps().getKeywordState(login, autotargetingId),
                equalTo(StateEnum.ON)
        );
        api.as(apiUserLogin);
        try {
            api.userSteps.keywordsSteps().shouldGetResultOn(
                    Action.SUSPEND,
                    login,
                    new SuspendRequestMap().withSelectionCriteria(
                            new IdsCriteriaMap().withIds(autotargetingId)
                    ),
                    expectedResult
            );
        } catch (Api5JsonError jsonError) {
            // Java версия сервиса возвращает 54-ю ошибку на весь запрос см. DIRECT-77051
            if (expectedError == null) {
                throw new AssertionError("неожиданная ошибка", jsonError);
            } else {
                assertThat("совпал текст ошибки (JSON)",
                        jsonError, Api5JsonErrorMatcher.equalToIgnoreLocale(expectedError.toJsonError()));
            }
        }
    }
}

