package ru.yandex.autotests.directapi.keywords.update;

import com.google.common.collect.Sets;
import com.yandex.direct.api.v5.general.AutotargetingCategoriesEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsBaseRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.AutotargetingCategoryMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordUpdateMap;
import ru.yandex.autotests.directapi.model.api5.keywords.UpdateRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.banners.KeywordsSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Collections.emptySet;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@Features(KeywordsFeatures.UPDATE)
@Description("Проверка нового поля AutotargetingCategories в методе Keywords.update")
@Issue("https://st.yandex-team.ru/DIRECT-149649")
public class UpdateAutotargetingCategoriesTest {

    private static final String CLIENT = KeywordsLogins.SINGLE_CLIENT_ADD;

    private static Long campaignId;
    private static Long adGroupId;
    private static DirectJooqDbSteps jooqDbSteps;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepareCampaign() {
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT);
    }

    @Before
    public void prepareAdGroup() {
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Test
    public void emptyInDBBeforeEmptyInRequest() {
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
        ).get(0);

        BidsBaseRecord bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(id);
        bidsBaseRecord.setRelevanceMatchCategories("");
        jooqDbSteps.bidsBaseSteps().updateBidsBase(bidsBaseRecord);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap().withId(id)
                ),
                ExpectedResult.success(id)
        );

        bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(id);

        assertThat("категории автотаргетинга корректно сохранились в БД",
                KeywordsSteps.getAutotargetingCategoriesEnumFromDB(bidsBaseRecord.getRelevanceMatchCategories()),
                equalTo(Sets.newHashSet(AutotargetingCategoriesEnum.values())));
    }

    @Test
    public void fullInDBBeforeEmptyInRequest() {
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
        ).get(0);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap().withId(id)
                ),
                ExpectedResult.success(id)
        );

        BidsBaseRecord bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(id);

        assertThat("категории автотаргетинга корректно сохранились в БД",
                KeywordsSteps.getAutotargetingCategoriesEnumFromDB(bidsBaseRecord.getRelevanceMatchCategories()),
                equalTo(Sets.newHashSet(AutotargetingCategoriesEnum.values())));
    }

    @Test
    public void notFullInDBBeforeEmptyInRequest() {
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
                .withAutotargetingCategories(new AutotargetingCategoryMap()
                        .withCategory(AutotargetingCategoriesEnum.ACCESSORY)
                        .withValue(YesNoEnum.NO))
        ).get(0);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap().withId(id)
                ),
                ExpectedResult.success(id)
        );

        BidsBaseRecord bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(id);

        assertThat("категории автотаргетинга корректно сохранились в БД",
                KeywordsSteps.getAutotargetingCategoriesEnumFromDB(bidsBaseRecord.getRelevanceMatchCategories()),
                equalTo(Sets.newHashSet(AutotargetingCategoriesEnum.ALTERNATIVE, AutotargetingCategoriesEnum.BROADER,
                        AutotargetingCategoriesEnum.COMPETITOR, AutotargetingCategoriesEnum.EXACT)));
    }

    @Test
    public void notFullInDBBeforeOneYesInRequest() {
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
                .withAutotargetingCategories(new AutotargetingCategoryMap()
                        .withCategory(AutotargetingCategoriesEnum.ACCESSORY)
                        .withValue(YesNoEnum.NO))
        ).get(0);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(id)
                                .withAutotargetingCategories(new AutotargetingCategoryMap()
                                        .withCategory(AutotargetingCategoriesEnum.ACCESSORY)
                                        .withValue(YesNoEnum.YES))
                ),
                ExpectedResult.success(id)
        );

        BidsBaseRecord bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(id);

        assertThat("категории автотаргетинга корректно сохранились в БД",
                KeywordsSteps.getAutotargetingCategoriesEnumFromDB(bidsBaseRecord.getRelevanceMatchCategories()),
                equalTo(Sets.newHashSet(AutotargetingCategoriesEnum.values())));
    }

    @Test
    public void notFullInDBBeforeOneNoInRequest() {
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
                .withAutotargetingCategories(new AutotargetingCategoryMap()
                        .withCategory(AutotargetingCategoriesEnum.ACCESSORY)
                        .withValue(YesNoEnum.NO))
        ).get(0);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(id)
                                .withAutotargetingCategories(new AutotargetingCategoryMap()
                                        .withCategory(AutotargetingCategoriesEnum.EXACT)
                                        .withValue(YesNoEnum.NO))
                ),
                ExpectedResult.success(id)
        );

        BidsBaseRecord bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(id);

        assertThat("категории автотаргетинга корректно сохранились в БД",
                KeywordsSteps.getAutotargetingCategoriesEnumFromDB(bidsBaseRecord.getRelevanceMatchCategories()),
                equalTo(Sets.newHashSet(AutotargetingCategoriesEnum.ALTERNATIVE, AutotargetingCategoriesEnum.BROADER,
                        AutotargetingCategoriesEnum.COMPETITOR)));
    }

    @Test
    public void tryToOffAllCategories() {
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
        ).get(0);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(id)
                                .withAutotargetingCategories(new AutotargetingCategoryMap()
                                                .withCategory(AutotargetingCategoriesEnum.ACCESSORY)
                                                .withValue(YesNoEnum.NO),
                                        new AutotargetingCategoryMap()
                                                .withCategory(AutotargetingCategoriesEnum.EXACT)
                                                .withValue(YesNoEnum.NO),
                                        new AutotargetingCategoryMap()
                                                .withCategory(AutotargetingCategoriesEnum.ALTERNATIVE)
                                                .withValue(YesNoEnum.NO),
                                        new AutotargetingCategoryMap()
                                                .withCategory(AutotargetingCategoriesEnum.BROADER)
                                                .withValue(YesNoEnum.NO),
                                        new AutotargetingCategoryMap()
                                                .withCategory(AutotargetingCategoriesEnum.COMPETITOR)
                                                .withValue(YesNoEnum.NO))
                ),
                ExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetails.OFF_ALL_AUTOTARGETING_CATEGORIES_IS_NOT_ALLOWED))
        );
    }

    @Test
    public void tryToOffLastCategory() {
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
                .withAutotargetingCategories(new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.ACCESSORY)
                                .withValue(YesNoEnum.NO),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.ALTERNATIVE)
                                .withValue(YesNoEnum.NO),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.EXACT)
                                .withValue(YesNoEnum.NO),
                        new AutotargetingCategoryMap()
                                .withCategory(AutotargetingCategoriesEnum.BROADER)
                                .withValue(YesNoEnum.NO))
        ).get(0);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(id)
                                .withAutotargetingCategories(new AutotargetingCategoryMap()
                                        .withCategory(AutotargetingCategoriesEnum.COMPETITOR)
                                        .withValue(YesNoEnum.NO))
                ),
                ExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetails.OFF_ALL_AUTOTARGETING_CATEGORIES_IS_NOT_ALLOWED))
        );
    }

    @Test
    public void emptyAutotargetingCategoriesAndInvalidType() {
        Long mobileContentCampaignId = api.userSteps.campaignSteps().addDefaultMobileAppCampaign(CLIENT);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultMobileGroup(mobileContentCampaignId);
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
        ).get(0);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap().withId(id)
                ),
                ExpectedResult.success(id)
        );

        BidsBaseRecord bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(id);

        assertThat("категории автотаргетинга корректно сохранились в БД",
                KeywordsSteps.getAutotargetingCategoriesEnumFromDB(bidsBaseRecord.getRelevanceMatchCategories()),
                equalTo(emptySet()));
    }

    @Test
    public void autotargetingCategoriesAndInvalidType() {
        Long mobileContentCampaignId = api.userSteps.campaignSteps().addDefaultMobileAppCampaign(CLIENT);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultMobileGroup(mobileContentCampaignId);
        Long id = api.userSteps.keywordsSteps().keywordsAdd(new KeywordAddMap()
                .shortAutotargeting(adGroupId)
        ).get(0);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(id)
                                .withAutotargetingCategories(new AutotargetingCategoryMap()
                                        .withCategory(AutotargetingCategoriesEnum.ACCESSORY)
                                        .withValue(YesNoEnum.NO))
                ),
                ExpectedResult.errors(new Notification(6001,
                        Api5ErrorDetails.AUTOTARGETING_CATEGORIES_FOR_THIS_AD_GROUP_ARE_NOT_ALLOWED))
        );
    }
}
