package ru.yandex.autotests.directapi.keywords.update;

import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordGetItemMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordUpdateMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;

import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by semkagtn on 07.05.15.
 */
@Aqua.Test
@Features(KeywordsFeatures.UPDATE)
@Issue("https://st.yandex-team.ru/DIRECT-36383")
@Description("Создание нового ключевого слова при обновлении текста старого ключевого слова")
@Tag(TagDictionary.TRUNK)
public class UpdateKeywordsCreateNewKeywordTest {

    private static final String LOGIN = KeywordsLogins.SINGLE_CLIENT_UPDATE;

    public static final String OLD_KEYWORD = "Старое ключевое слово";
    public static final String NEW_KEYWORD = "Новое ключевое слово";

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @Rule
    public Trashman trashman = new Trashman(api);

    private static Long campaingID;

    private long adGroupID;
    private long keywordID;

    @BeforeClass
    public static void createCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        campaingID = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createKeyword() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaingID);
        keywordID = api.userSteps.keywordsSteps().addKeyword(adGroupID, OLD_KEYWORD);
    }

    @Test
    @Description("Обновить текст в ключевом слове")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3751")
    public void checkNewKeywordId() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsUpdate(new KeywordUpdateMap()
                .withId(keywordID)
                .withKeyword(NEW_KEYWORD));
        assumeThat("успешно вернулся один идентификатор ключевого слова", keywordIDs, contains(notNullValue()));
    }

    @Test
    @Description("Проверить, что новое ключевое слово заменило в группе старое")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3752")
    public void checkIfNewKeywordReplacOldInAdGroup() {
        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsUpdate(new KeywordUpdateMap()
                .withId(keywordID)
                .withKeyword(NEW_KEYWORD));
        assumeThat("вернулся один идентификатор ключевого слова", keywordIDs, hasSize(1));
        long newKeywordID = keywordIDs.get(0);

        List<KeywordGetItem> response = api.userSteps.keywordsSteps().keywordsGet(new KeywordsSelectionCriteriaMap()
                .withAdGroupIds(adGroupID));
        assertThat("в группе нровое ключевое слово сменило старое", response, beanEquivalent(Arrays.asList(
                (KeywordGetItem) new KeywordGetItemMap()
                        .withId(newKeywordID).getBean())));
    }

    @Test
    @Description("Проверить, что вернулся уже существующий идентификатор ключевого слова," +
            " если при обновлении был указан текст старого ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3753")
    public void updateKeywordWithOldKeyword() {
        long anotherKeywordID = api.userSteps.keywordsSteps().addKeyword(adGroupID, NEW_KEYWORD);

        List<Long> keywordIDs = api.userSteps.keywordsSteps().keywordsUpdate(new KeywordUpdateMap()
                .withId(keywordID)
                .withKeyword(NEW_KEYWORD));
        assumeThat("вернулся один идентификатор ключевого слова", keywordIDs, hasSize(1));
        long newKeywordID = keywordIDs.get(0);
        assertThat("вернулся существующий идентификатор ключевого слова", newKeywordID, equalTo(anotherKeywordID));
    }
}
