package ru.yandex.autotests.directapi.keywords.update;

import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.keywords.KeywordFieldEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordGetItemMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordUpdateMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.keywords.UpdateRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by semkagtn on 30.04.15.
 * https://st.yandex-team.ru/TESTIRT-4580
 */
@Aqua.Test
@Features(KeywordsFeatures.UPDATE)
@Issue("https://st.yandex-team.ru/DIRECT-36383")
@Description("Одновременное обновление нескольких ключевых слов")
@Tag(TagDictionary.TRUNK)
public class UpdateKeywordsMultipleTest {

    private static final String LOGIN = KeywordsLogins.SINGLE_CLIENT_UPDATE;
    private static final String KEYWORD1 = "Баран жевал траву";
    private static final String KEYWORD2 = "Шла Саша по шоссе и сосала сушку";
    private static final String KEYWORD3 = "Ехал грека через реку";
    private static final String KEYWORD4 = "тур в Ватикан из Москвы";
    private static final String KEYWORD5 = "турчики в Ватикан из Москвы";
    private static final String KEYWORD6 = "туры к Папе в гости";
    private static final String KEYWORD7 = "дивный новый мир";

    private static final String[] KEYWORDS_WITH_SAME_NORMAL_FORM =
            {"тур в Ватикан из Москвы", "туры в Ватикан из Москвы"};

    private static Long campaignID;

    private long adGroupID;
    private long keywordID1;
    private long keywordID2;
    private long keywordID4;
    private long keywordID5;
    private long keywordID6;
    private Long autotargetingId;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @Rule
    public Trashman trashman = new Trashman(api);

    @BeforeClass
    public static void prepareCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createKeywords() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        keywordID1 = api.userSteps.keywordsSteps().addKeyword(adGroupID, KEYWORD1);
        keywordID2 = api.userSteps.keywordsSteps().addKeyword(adGroupID, KEYWORD2);
        keywordID4 = api.userSteps.keywordsSteps().addKeyword(adGroupID, KEYWORD4);
        keywordID5 = api.userSteps.keywordsSteps().addKeyword(adGroupID, KEYWORD5);
        keywordID6 = api.userSteps.keywordsSteps().addKeyword(adGroupID, KEYWORD6);
        autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupID);
    }

    @Test
    @Description("Обновить два ключевых слова одним и тем же текстом")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3760")
    public void updateTwoKeywordsWithSameText() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(keywordID1)
                                .withKeyword(KEYWORD1),
                        new KeywordUpdateMap()
                                .withId(keywordID2)
                                .withKeyword(KEYWORD1)),
                ExpectedResult.success(keywordID1),
                ExpectedResult.warnings(keywordID1, new Notification(10140)));
    }

    @Test
    @Description("Обновить два ключевых слова фразами с одинаковыми нормальными формами")
    public void updateTwoKeywordsByPhraseWithSameNormalForm() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(keywordID4)
                                .withKeyword(KEYWORDS_WITH_SAME_NORMAL_FORM[0]),
                        new KeywordUpdateMap()
                                .withId(keywordID5)
                                .withKeyword(KEYWORDS_WITH_SAME_NORMAL_FORM[1])),
                ExpectedResult.success(keywordID4),
                ExpectedResult.warnings(keywordID4, new Notification(10140)));
    }

    @Test
    @Description("Обновить два ключевых слова так, что в результате появится одна новая фраза")
    public void updateTwoKeywordsBySameNewPhrase() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(keywordID4)
                                .withKeyword(KEYWORD7),
                        new KeywordUpdateMap()
                                .withId(keywordID6)
                                .withKeyword(KEYWORD7)),
                JavaOrPerlExpectedResult
                        .java(ExpectedResult.success())
                        .perl(ExpectedResult.warnings(new Notification(10141))),
                JavaOrPerlExpectedResult.both(
                        ExpectedResult.warnings(new Notification(10140))));
    }

    @Test
    @Description("Два одинаковых идентификатора в запросе на обновление ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3759")
    public void sameKeywordIdsInRequest() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(keywordID1),
                        new KeywordUpdateMap()
                                .withId(keywordID1)),
                ExpectedResult.errors(new Notification(9800, Api5ErrorDetails.DUPLICATE_KEYWORD_IDS)),
                ExpectedResult.errors(new Notification(9800, Api5ErrorDetails.DUPLICATE_KEYWORD_IDS)));
    }

    @Test
    @Description("Обновить два ключевых слова в одном запросе")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3761")
    public void updateMultipleKeywordsInOneRequestTest() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(keywordID1),
                        new KeywordUpdateMap()
                                .withId(keywordID2),
                        new KeywordUpdateMap()
                                .withId(autotargetingId)
                ),
                ExpectedResult.success(keywordID1),
                ExpectedResult.success(keywordID2),
                ExpectedResult.success(autotargetingId)
        );
    }

    @Test
    @Description("Обновить два ключевых слова из разных групп в одном запросе")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3758")
    public void updateMultipleKeywordsFromDifferentGroupsInOneRequestTest() {
        long anotherGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        long anotherGroupKeywordID = api.userSteps.keywordsSteps().addDefaultKeyword(anotherGroupID);
        Long anotherAutotargetingId = api.userSteps.keywordsSteps().addAutotargeting(anotherGroupID);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(keywordID1),
                        new KeywordUpdateMap()
                                .withId(anotherGroupKeywordID),
                        new KeywordUpdateMap()
                                .withId(autotargetingId),
                        new KeywordUpdateMap()
                                .withId(anotherAutotargetingId)
                ),
                ExpectedResult.success(keywordID1),
                ExpectedResult.success(anotherGroupKeywordID),
                ExpectedResult.success(autotargetingId),
                ExpectedResult.success(anotherAutotargetingId)
        );
    }

    @Test
    @Description("Запрос на обновление двух ключевых слов с ожиданием в ответе успеха и ошибки")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3763")
    public void keywordWithSuccessErrorTest() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(keywordID1),
                        new KeywordUpdateMap()
                                .withId(keywordID2)
                                .withKeyword("")),
                ExpectedResult.success(keywordID1),
                ExpectedResult.errors(new Notification(5000,
                        Api5ErrorDetails.NO_VALUE_IN_FIELD, capitalize(KeywordUpdateMap.KEYWORD))));
    }

    @Test
    @Description("Запрос на обновление двух ключевых слов с ожиданием в ответе успеха и предупреждения")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3762")
    public void keywordWithSuccessWarningTest() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap()
                                .withId(keywordID1),
                        new KeywordUpdateMap()
                                .withId(keywordID2)
                                .withKeyword(KEYWORD3)),
                JavaOrPerlExpectedResult.both(
                        ExpectedResult.success(keywordID1)),
                JavaOrPerlExpectedResult
                        .java(ExpectedResult.success())
                        .perl(ExpectedResult.warnings(new Notification(10141))));
    }

    @Test
    public void updateKeywordsInReverseOrder() {
        String userParamForKeyword1 = "Спецсимволы";
        String userParamForKeyword2 = "Латинские символы";

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(
                        new KeywordUpdateMap().withId(keywordID2).withUserParam1(userParamForKeyword2),
                        new KeywordUpdateMap().withId(keywordID1).withUserParam1(userParamForKeyword1)
                ),
                ExpectedResult.success(keywordID2), ExpectedResult.success(keywordID1)
        );

        List<KeywordGetItem> updatedKeywords = api.userSteps.keywordsSteps().keywordsGet(
                new GetRequestMap()
                        .withSelectionCriteria(new KeywordsSelectionCriteriaMap().withIds(keywordID1, keywordID2))
                        .withFieldNames(KeywordFieldEnum.ID, KeywordFieldEnum.KEYWORD, KeywordFieldEnum.USER_PARAM_1)
        );

        assertThat("ключевые слова обновились правильно", updatedKeywords, beanDifferV5(Arrays.asList(
                (KeywordGetItem) new KeywordGetItemMap()
                        .withId(keywordID1)
                        .withKeyword(KEYWORD1)
                        .withUserParam1(userParamForKeyword1).getBean(),
                (KeywordGetItem) new KeywordGetItemMap()
                        .withId(keywordID2)
                        .withKeyword(KEYWORD2)
                        .withUserParam1(userParamForKeyword2).getBean())));
    }

}
