package ru.yandex.autotests.directapi.keywords.update;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordUpdateMap;
import ru.yandex.autotests.directapi.model.api5.keywords.UpdateRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 30.04.15.
 * https://st.yandex-team.ru/TESTIRT-4580
 */
@Aqua.Test
@Features(KeywordsFeatures.UPDATE)
@Stories(ApiStories.UNITS)
@Issue("https://st.yandex-team.ru/DIRECT-36383")
@Description("Списание баллов при обновлении ключевых слов")
public class UpdateKeywordsUnitsTest {

    private static final String LOGIN = KeywordsLogins.UNITS_UPDATE_CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int ADD_COST = 20;
    private static final int ITEM_COST_ON_ADD = 2;
    private static final int ADD_FAULT_COST = 20;
    private static final int GENERAL_FAULT_COST = 50;

    private static long keywordID1;
    private static long keywordID2;

    @BeforeClass
    public static void createGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        Long clientCampaignID = api.as(LOGIN).userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        long groupID = api.userSteps.adGroupsSteps().addDefaultGroup(clientCampaignID);
        keywordID1 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID);
        keywordID2 = api.userSteps.keywordsSteps().addDefaultKeyword(groupID);
    }

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    @Description("Обновление одного ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3771")
    public void updateSingleKeywordTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().keywordsUpdate(new KeywordUpdateMap()
                .withId(keywordID1));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assertThat("списалось верное количество баллов", unitsBefore - unitsAfter,
                equalTo(ADD_COST + ITEM_COST_ON_ADD));
    }

    @Test
    @Description("Обновление двух ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3772")
    public void updateTwoKeywordsTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().keywordsUpdate(
                new KeywordUpdateMap()
                        .withId(keywordID1),
                new KeywordUpdateMap()
                        .withId(keywordID2));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assertThat("списалось верное количество баллов", unitsBefore - unitsAfter,
                equalTo(ADD_COST + ITEM_COST_ON_ADD * 2));
    }

    @Test
    @Description("Обновление одного невалидного ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3773")
    public void updateInvalidKeywordItemTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap()
                        .withKeywords(new KeywordUpdateMap()
                                .withId(keywordID1)
                                .withKeyword("")),
                ExpectedResult.errors(new Notification(5000,
                        Api5ErrorDetails.NO_VALUE_IN_FIELD, capitalize(KeywordUpdateMap.KEYWORD))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);

        assertThat("списалось верное количество баллов", unitsBefore - unitsAfter,
                equalTo(ADD_COST + ADD_FAULT_COST));
    }

    @Test
    @Description("Обновление двух невалидных ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3774")
    public void updateTwoInvalidKeywordsTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap()
                        .withKeywords(
                                new KeywordUpdateMap()
                                        .withId(keywordID1)
                                        .withKeyword(""),
                                new KeywordUpdateMap()
                                        .withId(keywordID2)
                                        .withKeyword("")),
                ExpectedResult.errors(new Notification(5000,
                        Api5ErrorDetails.NO_VALUE_IN_FIELD, capitalize(KeywordUpdateMap.KEYWORD))),
                ExpectedResult.errors(new Notification(5000,
                        Api5ErrorDetails.NO_VALUE_IN_FIELD, capitalize(KeywordUpdateMap.KEYWORD))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);

        assertThat("списалось верное количество баллов", unitsBefore - unitsAfter,
                equalTo(ADD_COST + ADD_FAULT_COST * 2));
    }

    @Test
    @Description("Обновление одного валидного и одного невалидного ключевых слов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3775")
    public void updateOneValidOneInvalidKeywordsTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap()
                        .withKeywords(
                                new KeywordUpdateMap()
                                        .withId(keywordID1),
                                new KeywordUpdateMap()
                                        .withId(keywordID2)
                                        .withKeyword("")),
                ExpectedResult.success(),
                ExpectedResult.errors(new Notification(5000,
                        Api5ErrorDetails.NO_VALUE_IN_FIELD, capitalize(KeywordUpdateMap.KEYWORD))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);

        assertThat("списалось верное количество баллов",
                unitsBefore - unitsAfter, equalTo(ADD_COST + ADD_FAULT_COST + ITEM_COST_ON_ADD));
    }

    @Test
    @Description("Общая ошибка при обновлении ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3776")
    public void generalErrorOnUpdateKeywordTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(),
                JavaOrPerlApi5Error
                        .java(
                                new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                                        path(AddRequestMap.KEYWORDS), 1))
                        .perl(
                                new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                                        path(AddRequestMap.KEYWORDS)))
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);

        assertThat("списалось верное количество баллов", unitsBefore - unitsAfter,
                equalTo(GENERAL_FAULT_COST));
    }

    @Test
    @Description("Недостаточное колличество баллов при добавлнии ключевого слова")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3777")
    public void notEnoughUnitsOnUpdateKeywordTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, unitsBefore);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("доступные баллы пользователя", unitsBefore, equalTo(0));

        api.userSteps.keywordsSteps().shouldGetKeywordErrorOn(
                Action.UPDATE,
                new UpdateRequestMap()
                        .withKeywords(new KeywordUpdateMap()
                                .withId(keywordID1)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
    }
}
