package ru.yandex.autotests.directapi.keywords.update.keyword;

import java.sql.Timestamp;
import java.time.LocalDateTime;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.enums.BannerType;
import ru.yandex.autotests.directapi.keywords.KeywordsFeatures;
import ru.yandex.autotests.directapi.keywords.KeywordsLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordUpdateMap;
import ru.yandex.autotests.directapi.model.api5.keywords.UpdateRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by lightelfik on 26.01.16.
 * https://st.yandex-team.ru/TESTIRT-10946
 */
@Aqua.Test
@Features(KeywordsFeatures.UPDATE)
@Description("Поверка изменений LastChange при изменении ключевой фразы")
@Issue("https://st.yandex-team.ru/DIRECT-60743")
public class UpdateKeywordsKeywordLastChangeTest {

    private static final String CLIENT = KeywordsLogins.SINGLE_CLIENT_UPDATE;
    private static final String KEYWORD = "белый пушистый медведь";
    private static final String NEW_KEYWORD = "белый медведь";

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(5).as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static DirectJooqDbSteps directJooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT);

    private static Long adGroupId;
    private static Long adId;
    private static Long campaignId;

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        adId = api.userSteps.adsSteps().addDefaultAd(adGroupId, BannerType.TEXT);
    }

    @Test
    public void checkLastChange() {
        Long keywordId = api.userSteps.keywordsSteps().addKeyword(adGroupId, KEYWORD);
        Timestamp lastChange = Timestamp.valueOf(LocalDateTime.now().minusDays(5).withNano(0));
        directJooqDbSteps.bannersSteps().setBannersLastChange(adId, lastChange);
        directJooqDbSteps.adGroupsSteps().setPhrasesLastChange(adGroupId, lastChange);
        directJooqDbSteps.campaignsSteps().setCampaignsLastChange(campaignId, lastChange);

        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.UPDATE,
                new UpdateRequestMap().withKeywords(new KeywordUpdateMap()
                        .withId(keywordId)
                        .withKeyword(NEW_KEYWORD)),
                JavaOrPerlExpectedResult
                        .java(ExpectedResult.success())
                        .perl(ExpectedResult.warnings(new Notification(10141)))
        );

        Timestamp newAdLastChange = directJooqDbSteps.bannersSteps().getBanner(adId).getLastchange();
        Timestamp newAdGroupLastChange = directJooqDbSteps.adGroupsSteps().getPhrases(adGroupId).getLastchange();

        assumeThat("LastChange у объявления не изменился", newAdLastChange, equalTo(lastChange));
        assumeThat("LastChange у кампании не изменился", newAdLastChange, equalTo(lastChange));
        assertThat("LastChange у группы объявлений изменился", newAdGroupLastChange, greaterThan(lastChange));
    }

}
