package ru.yandex.autotests.directapi.keywordsresearch.deduplicate;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.keywordsresearch.DeduplicateErrorItem;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponse;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.keywordsresearch.KeywordsResearchFeatures;
import ru.yandex.autotests.directapi.keywordsresearch.KeywordsResearchLogins;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.DeduplicateRequestItemMap;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.DeduplicateRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;

@Aqua.Test
@Features(KeywordsResearchFeatures.DEDUPLICATE)
@Description("Негативные запросы с минус-словами")
@RunWith(Parameterized.class)
public class DeduplicateKeywordValidationNegativeTest {

    private static final int MAX_NUMBER_OF_WORDS = 7;
    private static final int WORD_MAX_LENGTH = 35;

    private static final String CLIENT = KeywordsResearchLogins.CLIENT;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(CLIENT);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public String keyword;

    @Parameterized.Parameter(value = 2)
    public Notification notification;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection fields() {
        Object[][] data = new Object[][] {
                {
                        "Ключевые слова содержат минус-слово", "доставка пицца -пицца",
                        new Notification(5161,
                                Api5ErrorDetailsJava.MINUS_WORD_CANT_SUBTRACT_PLUS_WORDS_IN_KEYWORDS, "пицца")
                },
                {
                        "Ключевые слова содержат минус-слово", "пицца -пицца",
                        new Notification(5161,
                                Api5ErrorDetailsJava.MINUS_WORD_CANT_SUBTRACT_PLUS_WORDS_IN_KEYWORDS, "пицца")
                },
                {
                        "Ключевое слово предваряется минус-словом", "высокий -полет квадрокоптер",
                        new Notification(5161,
                                Api5ErrorDetailsJava.ONLY_SEPARATE_MINUS_WORDS_CAN_BE_EXCLUDED_FROM_KEYWORDS)
                },
                {
                        "Минус-слово без ключевого", "-балкон",
                        new Notification(5161,
                                Api5ErrorDetailsJava.KEYWORDS_COULD_NOT_CONSIST_OF_NEGATIVE_KEYWORDS_ONLY)
                },
                {
                        "Простая группировка с минус-словами", "сервисное обслуживание (Kia | Ford) -гарантийное -бесплатное",
                        new Notification(5002, Api5ErrorDetailsJava.INVALID_CHARS,
                                path(DeduplicateRequestItemMap.KEYWORD))
                },
                {
                        "2 точки в минус-слове", "пицца -57.21.45",
                        new Notification(5161,
                                Api5ErrorDetailsJava.ONLY_SEPARATE_WORDS_CAN_BE_EXCLUDED_FROM_KEYWORDS_PHRASES)
                },
                {
                        "Минус-слово в кавычках", "\"-пицца\"",
                        new Notification(5161, Api5ErrorDetailsJava.BOTH_QUOTES_AND_MINUS_WORDS_IN_KEYWORDS)
                },
                {
                        "Неправильное использование специальных символов", "[.пицца -доставка",
                        new Notification(5002,
                                Api5ErrorDetailsJava.NO_MODIFIERS_INSIDE_SQUARE_BRACKETS_IN_KEYWORDS)
                },
                {
                        "Неправильное использование специальных символов", "пицца -'доставка -гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "-")
                },
                {
                        "Минус-слово с точкой", "гостиница -пиц.ца -доставка",
                        new Notification(5161,
                                Api5ErrorDetailsJava.ONLY_SEPARATE_WORDS_CAN_BE_EXCLUDED_FROM_KEYWORDS_PHRASES)
                },
                {
                        "Неправильное использование знака +", "пицца доставка++отель гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "+")
                },
                {
                        "Неправильное использование знака +", "пицца доставка+ гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "+")
                },
                {
                        "Неправильное использование знака +", "пицца доставка гостиница+",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "+")
                },
                {
                        "Неправильное использование знака +", "пицца доставка+гостиница отель",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "+")
                },
                {
                        "Неправильное использование знака -", "пицца отель--гостиница доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "-")
                },
                {
                        "Неправильное использование знака -", "пицца доставка- гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "-")
                },
                {
                        "Неправильное использование знака -", "пицца доставка отель-",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "-")
                },
                {
                        "Неправильное использование знака !", "пицца отель !-доставка гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "!")
                },
                {
                        "Неправильное использование знака !", "пицца отель X-!доставка гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "!")
                },
                {
                        "Неправильное использование знака !", "пицца доставка -!-отель гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "!")
                },
                {
                        "Неправильное использование знака +", "гостиница отель +-пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "+")
                },
                {
                        "Неправильное использование знака +", "гостиница отель X-+пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "+")
                },
                {
                        "Неправильное использование знака +", "гостиница отель -+-пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "+")
                },
                {
                        "Восклицательный знак с другим модификатором", "гостиница !\"отель-пицца\" доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "!")
                },
                {
                        "Восклицательный знак в конце фразы", "гостиница отель-пицца доставка!",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "!")
                },
                {
                        "Восклицательный знак без слова", "гостиница ! отель-пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "!")
                },
                {
                        "Два восклицательных знака подряд", "гостиница !!отель-пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "!")
                },
                {
                        "Знак минус внутри квадратных скобок", "пицца [-отель] доставка",
                        new Notification(5002,
                                Api5ErrorDetailsJava.NO_MODIFIERS_INSIDE_SQUARE_BRACKETS_IN_KEYWORDS)
                },
                {
                        "Модификаторы внутри квадратных скобок", "пицца [+\"отель\"] доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORDS, "+")
                },
                {
                        "Вложенные квадратные скобки", "пицца [ [отель] доставка]",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORDS)
                },
                {
                        "Пустые квадратные скобки", "пицца [] отель доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORDS)
                },
                {
                        "Квадратные скобки в неправильном порядке", "]пицца[",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORDS)
                },
                {
                        "Неправильная скобочная последовательность для квадратных скобок", "пицца [отель [доставка гостиница]",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORDS)
                },
                {
                        "Неправильная скобочная последовательность для квадратных скобок", "пицца гостиница] отель",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORDS)
                },
                {
                        "Неправильная скобочная последовательность для квадратных скобок", "пицца [гостиница отель",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORDS)
                },
                {
                        "8 ключевых слов", "пицца отель москва гостиница поезд самолет такси теплоход",
                        new Notification(5140, Api5ErrorDetailsJava.KEYWORDS_COULD_NOT_CONTAIN_MORE_WORDS,
                                MAX_NUMBER_OF_WORDS)
                },
                {
                        "Слишком длинное ключевое слово", "отель пииииииииииииииииииииииииииииииииииииииииииицца теплоход",
                        new Notification(5142, Api5ErrorDetailsJava.KEYWORD_MAX_LENGTH_EXCEEDED_INDIVIDUAL,
                                WORD_MAX_LENGTH)
                },
                {
                        "3 минус-слова без ключевых", "-доставка -пицца -москва",
                        new Notification(5161,
                                Api5ErrorDetailsJava.KEYWORDS_COULD_NOT_CONSIST_OF_NEGATIVE_KEYWORDS_ONLY)
                },
                {
                        "3 стоп слова без ключевых", "в из за",
                        new Notification(5141, Api5ErrorDetailsJava.KEYWORDS_COULD_NOT_CONSIST_OF_STOPWORDS_ONLY)
                },
                {
                        "Стоп-слово без ключевых", "в",
                        new Notification(5141, Api5ErrorDetailsJava.KEYWORDS_COULD_NOT_CONSIST_OF_STOPWORDS_ONLY)
                },
                {
                        "Ключевая фраза с тремя кавычками", "\"пицца доставка\" гостиница\"",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_QUOTES_IN_KEYWORDS)
                },
                {
                        "Ключевая фраза с одной кавычкой", "\"пицца",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_QUOTES_IN_KEYWORDS)
                },
                {
                        "Ключевая фраза с точкой в конце", "пицца .",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_DOT_IN_KEYWORDS)
                },
                {
                        "Ключевая фраза с точкой между словами", "пицца . доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_DOT_IN_KEYWORDS)
                },
                {
                        "Ключевая фраза с точкой в начале", ". пицца",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_DOT_IN_KEYWORDS)
                },
                {
                        "Ключевая фраза, содержащая невалидный символ", "aaaaa\uE751bbbbbb",
                        new Notification(5002, Api5ErrorDetailsJava.INVALID_CHARS,
                                path(DeduplicateRequestItemMap.KEYWORD))
                },
        };
        return Arrays.asList(data);
    }

    @Test
    public void checkDeduplicate() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withKeywords(new DeduplicateRequestItemMap().withKeyword(keyword));

        DeduplicateResponse response =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(CLIENT, request);

        assertThat("Ответ содержит одну ошибку", response.getFailure(), hasSize(1));

        DeduplicateErrorItem errorItem = response.getFailure().get(0);

        assertThat("Ответ содержит одну ошибку", errorItem.getErrors(), hasSize(1));
        assertThat("Ответ содержит правильную ошибку", errorItem.getErrors().get(0),
                is(notification.getBean()));
    }
}
