package ru.yandex.autotests.directapi.keywordsresearch.hassearchvolume;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.keywordsresearch.KeywordsResearchFeatures;
import ru.yandex.autotests.directapi.keywordsresearch.KeywordsResearchLogins;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.HasSearchVolumeRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.HasSearchVolumeSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeFieldEnum.ALL_DEVICES;
import static com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeFieldEnum.DESKTOPS;
import static com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeFieldEnum.KEYWORD;
import static com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeFieldEnum.MOBILE_PHONES;
import static com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeFieldEnum.TABLETS;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;

@Aqua.Test
@Features(KeywordsResearchFeatures.HAS_SEARCH_VOLUME)
@Description("Негативные запросы с минус-словами")
@RunWith(Parameterized.class)
public class HasSearchVolumeKeywordValidationNegativeTest {

    private static final int MAX_NUMBER_OF_WORDS = 7;
    private static final int WORD_MAX_LENGTH = 35;
    private static final int THREE_SECONDS = 3 * 1000;

    private static final String CLIENT = KeywordsResearchLogins.CLIENT;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(CLIENT);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public String keyword;

    @Parameterized.Parameter(value = 2)
    public Api5Error error;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection fields() {
        Object[][] data = new Object[][] {
                {
                        "Ключевые слова содержат минус-слово", "доставка пицца -пицца",
                        new Api5Error(5161, Api5ErrorDetailsJava.MINUS_WORD_CANT_SUBTRACT_PLUS_WORDS,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "Ключевые слова содержат минус-слово", "пицца -пицца",
                        new Api5Error(5161, Api5ErrorDetailsJava.MINUS_WORD_CANT_SUBTRACT_PLUS_WORDS,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "Ключевое слово предваряется минус-словом", "высокий -полет квадрокоптер",
                        new Api5Error(5161, Api5ErrorDetailsJava.ONLY_SEPARATE_MINUS_WORDS_CAN_BE_EXCLUDED_FROM_KEYWORD,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "Минус-слово без ключевого", "-балкон",
                        new Api5Error(5161, Api5ErrorDetailsJava.KEYWORD_COULD_NOT_CONSIST_OF_NEGATIVE_KEYWORDS_ONLY,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "Простая группировка с минус-словами", "сервисное обслуживание (Kia | Ford) -гарантийное -бесплатное",
                        new Api5Error(5002, Api5ErrorDetailsJava.INVALID_CHARS,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "2 точки в минус-слове", "пицца -57.21.45",
                        new Api5Error(5161, Api5ErrorDetailsJava.ONLY_SEPARATE_WORDS_CAN_BE_EXCLUDED_FROM_KEYWORD_PHRASES,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "Минус-слово в кавычках", "\"-пицца\"",
                        new Api5Error(5161, Api5ErrorDetailsJava.BOTH_QUOTES_AND_MINUS_WORDS_IN_KEYWORD,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "Неправильное использование специальных символов", "[.пицца -доставка",
                        new Api5Error(5002, Api5ErrorDetailsJava.NO_MODIFIERS_INSIDE_SQUARE_BRACKETS,
                                "[.пицца -доставка")
                },
                {
                        "Неправильное использование специальных символов", "пицца -'доставка -гостиница",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "пицца -'доставка -гостиница", "-")
                },
                {
                        "Минус-слово с точкой", "гостиница -пиц.ца -доставка",
                        new Api5Error(5161, Api5ErrorDetailsJava.ONLY_SEPARATE_WORDS_CAN_BE_EXCLUDED_FROM_KEYWORD_PHRASES,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "Неправильное использование знака +", "пицца доставка++отель гостиница",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "пицца доставка++отель гостиница", "+")
                },
                {
                        "Неправильное использование знака +", "пицца доставка+ гостиница",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "пицца доставка+ гостиница", "+")
                },
                {
                        "Неправильное использование знака +", "пицца доставка гостиница+",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "пицца доставка гостиница+", "+")
                },
                {
                        "Неправильное использование знака +", "пицца доставка+гостиница отель",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "пицца доставка+гостиница отель", "+")
                },
                {
                        "Неправильное использование знака -", "пицца отель--гостиница доставка",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "пицца отель--гостиница доставка", "-")
                },
                {
                        "Неправильное использование знака -", "пицца доставка- гостиница",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "пицца доставка- гостиница", "-")
                },
                {
                        "Неправильное использование знака -", "пицца доставка отель-",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "пицца доставка отель-", "-")
                },
                {
                        "Неправильное использование знака !", "пицца отель !-доставка гостиница",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "пицца отель !-доставка гостиница", "!")
                },
                {
                        "Неправильное использование знака !", "пицца отель X-!доставка гостиница",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "пицца отель X-!доставка гостиница", "!")
                },
                {
                        "Неправильное использование знака !", "пицца доставка -!-отель гостиница",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "пицца доставка -!-отель гостиница", "!")
                },
                {
                        "Неправильное использование знака +", "гостиница отель +-пицца доставка",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "гостиница отель +-пицца доставка", "+")
                },
                {
                        "Неправильное использование знака +", "гостиница отель X-+пицца доставка",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "гостиница отель X-+пицца доставка", "+")
                },
                {
                        "Неправильное использование знака +", "гостиница отель -+-пицца доставка",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "гостиница отель -+-пицца доставка", "+")
                },
                {
                        "Восклицательный знак с другим модификатором", "гостиница !\"отель-пицца\" доставка",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "гостиница !\"отель-пицца\" доставка", "!")
                },
                {
                        "Восклицательный знак в конце фразы", "гостиница отель-пицца доставка!",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "гостиница отель-пицца доставка!", "!")
                },
                {
                        "Восклицательный знак без слова", "гостиница ! отель-пицца доставка",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "гостиница ! отель-пицца доставка", "!")
                },
                {
                        "Два восклицательных знака подряд", "гостиница !!отель-пицца доставка",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "гостиница !!отель-пицца доставка", "!")
                },
                {
                        "Знак минус внутри квадратных скобок", "пицца [-отель] доставка",
                        new Api5Error(5002, Api5ErrorDetailsJava.NO_MODIFIERS_INSIDE_SQUARE_BRACKETS,
                                "пицца [-отель] доставка")
                },
                {
                        "Модификаторы внутри квадратных скобок", "пицца [+\"отель\"] доставка",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_SYMBOL_IN_KEYWORD,
                                "пицца [+\"отель\"] доставка", "+")
                },
                {
                        "Вложенные квадратные скобки", "пицца [ [отель] доставка]",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORD,
                                "пицца [ [отель] доставка]")
                },
                {
                        "Пустые квадратные скобки", "пицца [] отель доставка",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORD,
                                "пицца [] отель доставка")
                },
                {
                        "Квадратные скобки в неправильном порядке", "]пицца[",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORD,
                                "]пицца[")
                },
                {
                        "Неправильная скобочная последовательность для квадратных скобок", "пицца [отель [доставка гостиница]",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORD,
                                "пицца [отель [доставка гостиница]")
                },
                {
                        "Неправильная скобочная последовательность для квадратных скобок", "пицца гостиница] отель",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORD,
                                "пицца гостиница] отель")
                },
                {
                        "Неправильная скобочная последовательность для квадратных скобок", "пицца [гостиница отель",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORD,
                                "пицца [гостиница отель")
                },
                {
                        "8 ключевых слов", "пицца отель москва гостиница поезд самолет такси теплоход",
                        new Api5Error(5140, Api5ErrorDetailsJava.KEYWORD_COULD_NOT_CONTAIN_MORE_WORDS,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0), MAX_NUMBER_OF_WORDS)
                },
                {
                        "Слишком длинное ключевое слово", "отель пииииииииииииииииииииииииииииииииииииииииииицца теплоход",
                        new Api5Error(5142, Api5ErrorDetailsJava.KEYWORD_MAX_LENGTH_EXCEEDED_INDIVIDUAL,
                                WORD_MAX_LENGTH)
                },
                {
                        "3 минус-слова без ключевых", "-доставка -пицца -москва",
                        new Api5Error(5161, Api5ErrorDetailsJava.KEYWORD_COULD_NOT_CONSIST_OF_NEGATIVE_KEYWORDS_ONLY,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "3 стоп слова без ключевых", "в из за",
                        new Api5Error(5141, Api5ErrorDetailsJava.KEYWORD_COULD_NOT_CONSIST_OF_STOPWORDS_ONLY,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "Стоп-слово без ключевых", "в",
                        new Api5Error(5141, Api5ErrorDetailsJava.KEYWORD_COULD_NOT_CONSIST_OF_STOPWORDS_ONLY,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "Ключевая фраза с тремя кавычками", "\"пицца доставка\" гостиница\"",
                        new Api5Error(5002, Api5ErrorDetailsJava.KEYWORD_INCORRECT_USAGE_OF_QUOTES,
                                "\"пицца доставка\" гостиница\"")
                },
                {
                        "Ключевая фраза с одной кавычкой", "\"пицца",
                        new Api5Error(5002, Api5ErrorDetailsJava.KEYWORD_INCORRECT_USAGE_OF_QUOTES,
                                "\"пицца")
                },
                {
                        "Ключевая фраза с точкой в конце", "пицца .",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_DOT_IN_KEYWORD,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "Ключевая фраза с точкой между словами", "пицца . доставка",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_DOT_IN_KEYWORD,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "Ключевая фраза с точкой в начале", ". пицца",
                        new Api5Error(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_DOT_IN_KEYWORD,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
                {
                        "Ключевая фраза, содержащая невалидный символ", "aaaaa\uE751bbbbbb",
                        new Api5Error(5002, Api5ErrorDetailsJava.INVALID_CHARS,
                                path(HasSearchVolumeSelectionCriteriaMap.KEYWORDS, 0))
                },
        };
        return Arrays.asList(data);
    }

    @Before
    public void delayRequest() throws InterruptedException {
        // Разрешено не более 20 запросов за окно в 60 секунд
        Thread.sleep(THREE_SECONDS);
    }

    @Test
    public void checkHasSearchVolume() {
        HasSearchVolumeRequestMap request = new HasSearchVolumeRequestMap()
                .withSelectionCriteria(new HasSearchVolumeSelectionCriteriaMap()
                        .withKeywords(keyword)
                        .withRegionIds(0L))
                .withFieldNames(KEYWORD, DESKTOPS, MOBILE_PHONES, TABLETS, ALL_DEVICES);

        api.userSteps.keywordsResearchSteps()
                .shouldGetErrorOnKeywordsResearchHasSearchVolume(request, CLIENT, error);
    }
}
