package ru.yandex.autotests.directapi.leads.get;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;

import com.google.common.collect.Iterables;
import com.yandex.direct.api.v5.leads.GetRequest;
import com.yandex.direct.api.v5.leads.GetResponse;
import com.yandex.direct.api.v5.leads.LeadFieldEnum;
import com.yandex.direct.api.v5.leads.LeadsSelectionCriteria;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.beans.JsonObject;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.leads.LeadsFeatures;
import ru.yandex.autotests.directapi.model.api5.leads.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.httpclient.lite.core.config.HttpClientConnectionConfig;
import ru.yandex.autotests.httpclient.lite.core.config.HttpClientFactory;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.directapi.leads.LeadsLogins.LEADS_CLIENT;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@Features(LeadsFeatures.GET)
@Description("Тест на общую корректность работы сервиса")
@Tag(TagDictionary.TRUNK)
public class GetPositiveTest {
    private static final Logger logger = LoggerFactory.getLogger(GetPositiveTest.class);
    private static final String client = LEADS_CLIENT;
    private static final Long TURBO_PAGE_ID = 1000833L;

    // https://a.yandex-team.ru/arc/trunk/arcadia/canvas/landing-form-backend/ansible/dev_playbook.yml?rev=6914473&blame=true
    // Настройка ENCRYPTION_TOKEN
    private static final String ENCRYPTION_TOKEN = "s3cr3t!";
    // Есть вероятность, что скоро поменяется урл
    private static final String UDATA_URL = "https://dev-turbofeedback.common.yandex.ru/udata";

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @Rule
    public Trashman trashman = new Trashman(api);

    @BeforeClass
    public static void init() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
    }

    @Test
    public void smokeTest() throws IOException {
        GetRequest request = new GetRequest();

        LeadsSelectionCriteria selectionCriteria = new LeadsSelectionCriteria();
        selectionCriteria.setTurboPageIds(singletonList(TURBO_PAGE_ID));
        request.setSelectionCriteria(selectionCriteria);

        request.setFieldNames(asList(LeadFieldEnum.ID, LeadFieldEnum.DATA, LeadFieldEnum.SUBMITTED_AT,
                LeadFieldEnum.TURBO_PAGE_ID, LeadFieldEnum.TURBO_PAGE_NAME));

        GetResponse response = api.userSteps.leadsSteps().leadsGetRawResponse(client, new GetRequestMap(request));
        if (response.getLeads().isEmpty()) {
            logger.info("Get leads response is empty, trying to post some landing page");
            postLandingPage();
            response = api.userSteps.leadsSteps().leadsGetRawResponse(client, new GetRequestMap(request));
        }
        assertThat("Получен ответ без ошибок", response.getLeads(), not(empty()));
    }

    private void postLandingPage() throws IOException {
        DirectJooqDbSteps jooqSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(client);
        long clientId = jooqSteps.usersSteps().getUser(client).getClientid();

        JsonObject meta = new JsonObject()
                .with("form_id", String.valueOf(TURBO_PAGE_ID))
                .with("landing_id", String.valueOf(TURBO_PAGE_ID))
                .with("client_id", clientId)
                .with("emails", singletonList("nonexistent@example.com"))
                .with("validation", new JsonObject()
                        .with("required", new JsonObject())
                        .with("optional", new JsonObject()));

        JsonObject udata = new JsonObject()
                .with("form", String.valueOf(TURBO_PAGE_ID))
                .with("sk", "")
                .with("meta", encrypt(meta.toString(), ENCRYPTION_TOKEN));

        StringEntity entity = new StringEntity(udata.toString(), ContentType.APPLICATION_JSON);
        HttpPost httpRequest = new HttpPost(UDATA_URL);
        httpRequest.setEntity(entity);
        HttpClientConnectionConfig clientConnectionConfig =
                ConnectionConfig.getHttpClientConnectionConfig(UDATA_URL).timeout(360);
        CloseableHttpClient httpClient = HttpClientFactory.getHttpClient(clientConnectionConfig);
        CloseableHttpResponse response = httpClient.execute(httpRequest);
        if (response.getStatusLine().getStatusCode() != 200) {
            Assert.fail(String.format("Error while posting landing page. Status: %s, body: %s",
                    response.getStatusLine().getStatusCode(), IOUtils.toString(response.getEntity().getContent())));
        }
    }

    /**
     * https://a.yandex-team.ru/arc/trunk/arcadia/canvas/landing-form-backend/encryptor/encryptor.py?rev=6913265
     */
    private String encrypt(String data, String key) {
        byte[] dataBytes = data.getBytes(StandardCharsets.UTF_8);
        byte[] keyBytes = key.getBytes(StandardCharsets.UTF_8);
        Iterator<Byte> keyIterator = Iterables.cycle(ArrayUtils.toObject(keyBytes)).iterator();
        byte[] result = new byte[dataBytes.length];
        for (int i = 0; i < dataBytes.length; i++) {
            result[i] = (byte) (dataBytes[i] ^ keyIterator.next());
        }
        return Base64.encodeBase64String(result);
    }
}
