package ru.yandex.autotests.directapi.reports;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.reports.DateRangeTypeEnum;
import com.yandex.direct.api.v5.reports.FieldEnum;
import com.yandex.direct.api.v5.reports.FormatEnum;
import com.yandex.direct.api.v5.reports.ReportTypeEnum;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsData;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsLine;
import ru.yandex.autotests.directapi.model.api5.reports.SelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by onotole on 7/28/16.
 * https://st.yandex-team.ru/TESTIRT-9906
 */
@Aqua.Test
@Description("Проверка работы группировок")
@Issue("https://st.yandex-team.ru/TESTIRT-9244")
@Features(ReportsFeatures.ONLINE_CUSTOM_REPORT)
@RunWith(Parameterized.class)
public class GroupByTest {
    private static final String CLIENT = ReportsLogins.CLIENT_FOR_GROUP_BY;
    private static final String dateFrom = "2020-05-02";
    private static final String dateTo = "2020-05-07";
    private static int expectedClicks;
    private static BigDecimal expectedRevenue;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER).clientLogin(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public FieldEnum fieldForGrouping;

    @Parameterized.Parameter(1)
    public int expectedNumberOfLines;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][] {
                {FieldEnum.AGE, 7},
                {FieldEnum.AD_NETWORK_TYPE, 2},
                {FieldEnum.CARRIER_TYPE, 2},
                {FieldEnum.CLICK_TYPE, 2},
                {FieldEnum.CRITERIA_TYPE, 4},
                {FieldEnum.CRITERION_TYPE, 3},
                {FieldEnum.DEVICE, 3},
                {FieldEnum.GENDER, 3},
                {FieldEnum.AD_FORMAT, 4},
                {FieldEnum.MOBILE_PLATFORM, 3},
                {FieldEnum.SLOT, 2},
        });
    }

    @BeforeClass
    public static void prepare() {
        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withFieldNames(FieldEnum.CLICKS, FieldEnum.IMPRESSIONS, FieldEnum.REVENUE)
                .withDateRangeType(DateRangeTypeEnum.CUSTOM_DATE)
                .withUniqueReportName()
                .withReportType(ReportTypeEnum.CUSTOM_REPORT)
                .withFormat(FormatEnum.TSV)
                .withIncludeVAT(YesNoEnum.YES)
                .withIncludeDiscount(YesNoEnum.NO)
                .withSelectionCriteria(new SelectionCriteriaMap()
                        .withDateFrom(dateFrom)
                        .withDateTo(dateTo));
        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);
        assumeThat("вернулся репорт с хотя бы одной строкой", reportsData.getReportsLines().size(), greaterThan(0));

        ReportsLine reportsLine = reportsData.getReportsLines().get(0);
        expectedClicks = reportsLine.getClicks();
        expectedRevenue = reportsLine.getRevenue();
    }

    @Test
    public void checkGroupBy() {
        ReportDefinitionMap groupedReportDefinitionMap = new ReportDefinitionMap()
                .withFieldNames(FieldEnum.CLICKS, FieldEnum.REVENUE, fieldForGrouping)
                .withDateRangeType(DateRangeTypeEnum.CUSTOM_DATE)
                .withUniqueReportName()
                .withReportType(ReportTypeEnum.CUSTOM_REPORT)
                .withFormat(FormatEnum.TSV)
                .withIncludeVAT(YesNoEnum.YES)
                .withIncludeDiscount(YesNoEnum.NO)
                .withSelectionCriteria(new SelectionCriteriaMap()
                    .withDateFrom(dateFrom)
                    .withDateTo(dateTo));
        ReportsData groupedReportsData = api.userSteps.reportsSteps().callReportsXml(groupedReportDefinitionMap);
        assumeThat("вернулся репорт с ожидаемым количеством строк", groupedReportsData.getReportsLines().size(),
                equalTo(expectedNumberOfLines));

        List<ReportsLine> groupedReportsLine = groupedReportsData.getReportsLines();
        int clicks = 0;
        BigDecimal revenue = new BigDecimal(0);
        for (ReportsLine line: groupedReportsLine) {
            clicks += line.getClicks();
            revenue = revenue.add(line.getRevenue());
        }

        assertThat("после группировки количество кликов в репорте верное", expectedClicks, equalTo(clicks));
        assertThat("после группировки доход в репорте верный", expectedRevenue, equalTo(revenue));
    }

}
