package ru.yandex.autotests.directapi.reports;

import com.yandex.direct.api.v5.reports.*;
import org.junit.ClassRule;
import org.junit.Test;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.*;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by lightelfik on 06.03.17.
 */

@Aqua.Test
@Description("Проверяем что нельзя запросить отчеты с разными параметрами и одинаковыми названиями")
@Issue("https://st.yandex-team.ru/DIRECT-63290")
@Features(ReportsFeatures.OFFLINE_CUSTOM_REPORT)
public class NameTest {

    private static final String DATE = "2020-02-28";
    private static final Long CAMPAIGN_ID = 1287989L;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER).clientLogin(ReportsLogins.CLIENT_RUB_WITH_STAT3);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Test
    public void testUniqueName() {
        // проверяем что два отчета с разными FieldNames не могут иметь одинаковое ReportName
        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultCustomReportWithUniqueReportName()
                .withCustomDatesAndCampaignIds(DATE, DATE, CAMPAIGN_ID)
                .withFieldNames(FieldEnum.IMPRESSIONS);
        ReportsData reportsData = api.processingMode(ReportProcessingMode.OFFLINE).userSteps
                .reportsSteps().callReportsXml(reportDefinitionMap);
        assumeThat("вернулся непустой отчёт", reportsData.getReportsLines(), not(empty()));
        String name = reportsData.getReportHeaderName();
        ReportDefinitionMap newReportDefinitionMap = new ReportDefinitionMap()
                .withDefaultCustomReport()
                .withReportName(name)
                .withCustomDatesAndCampaignIds(DATE, DATE, CAMPAIGN_ID)
                .withFieldNames(FieldEnum.CLICKS);
        api.processingMode(ReportProcessingMode.OFFLINE).userSteps
                .reportsSteps().expectXmlErrorOnReports(newReportDefinitionMap,
                    new Api5Error(4000, Api5ErrorDetails.DIFFERENT_REPORT_WITH_SAME_NAME, "ReportName" ));
    }

    @Test
    public void testUniqueNameWithMoneyInMicrosHeader() {
        // проверяем что два отчета с одинаковыми параметрами, но разными значениями заголовка returnMoneyInMicros
        // не могут иметь одинаковое ReportName
        ReportsData reportsData = api.processingMode(ReportProcessingMode.OFFLINE).userSteps
                .reportsSteps().callReportsXml(getReportDefinitionMap());
        assumeThat("вернулся непустой отчёт", reportsData.getReportsLines(), not(empty()));
        api.processingMode(ReportProcessingMode.OFFLINE).returnMoneyInMicros(false).userSteps
                .reportsSteps().expectXmlErrorOnReports(
                    getReportDefinitionMap().withReportName(reportsData.getReportHeaderName()),
                    new Api5Error(4000, Api5ErrorDetails.DIFFERENT_REPORT_WITH_SAME_NAME, "ReportName" )
                );
        api.returnMoneyInMicros(null);
    }

    @Test
    public void testUniqueNameWithDifferentAcceptLanguage() {
        // проверяем что два отчета с одинаковыми параметрами, но разными значениями заголовка Accept-Language
        // не могут иметь одинаковое ReportName
        ReportsData reportsData = api.locale("ru").processingMode(ReportProcessingMode.OFFLINE).userSteps
                .reportsSteps().callReportsXml(getReportDefinitionMap());
        assumeThat("вернулся непустой отчёт", reportsData.getReportsLines(), not(empty()));
        api.locale("en").processingMode(ReportProcessingMode.OFFLINE).userSteps
                .reportsSteps().expectXmlErrorOnReports(
                    getReportDefinitionMap().withReportName(reportsData.getReportHeaderName()),
                    new Api5Error(4000, Api5ErrorDetails.DIFFERENT_REPORT_WITH_SAME_NAME, "ReportName" )
                );
    }

    @Test
    public void testUniqueNameWithSkipReportHeader() {
        ReportsData reportsData = api.processingMode(ReportProcessingMode.OFFLINE).userSteps
                .reportsSteps().callReportsXml(getReportDefinitionMap());
        assumeThat("вернулся непустой отчёт", reportsData.getReportsLines(), not(empty()));
        api.processingMode(ReportProcessingMode.OFFLINE).skipReportHeader(true).userSteps
                .reportsSteps().expectXmlErrorOnReports(
                    getReportDefinitionMap().withReportName(reportsData.getReportHeaderName()),
                    new Api5Error(4000, Api5ErrorDetails.DIFFERENT_REPORT_WITH_SAME_NAME, "ReportName" )
                );
        api.skipReportHeader(null);
    }

    @Test
    public void testUniqueNameWithSkipColumnHeader() {
        ReportsData reportsData = api.processingMode(ReportProcessingMode.OFFLINE).userSteps
                .reportsSteps().callReportsXml(getReportDefinitionMap());
        assumeThat("вернулся непустой отчёт", reportsData.getReportsLines(), not(empty()));
        String name = reportsData.getReportHeaderName();
        api.processingMode(ReportProcessingMode.OFFLINE).skipColumnHeader(true).userSteps
                .reportsSteps().expectXmlErrorOnReports(
                    getReportDefinitionMap().withReportName(reportsData.getReportHeaderName()),
                    new Api5Error(4000, Api5ErrorDetails.DIFFERENT_REPORT_WITH_SAME_NAME, "ReportName" )
                );
        api.skipColumnHeader(null);
    }

    @Test
    public void testUniqueNameWithSkipReportSummary() {
        ReportsData reportsData = api.processingMode(ReportProcessingMode.OFFLINE).userSteps
                .reportsSteps().callReportsXml(getReportDefinitionMap());
        assumeThat("вернулся непустой отчёт", reportsData.getReportsLines(), not(empty()));
        api.processingMode(ReportProcessingMode.OFFLINE).skipReportSummary(true).userSteps
                .reportsSteps().expectXmlErrorOnReports(
                    getReportDefinitionMap().withReportName(reportsData.getReportHeaderName()),
                    new Api5Error(4000, Api5ErrorDetails.DIFFERENT_REPORT_WITH_SAME_NAME, "ReportName" )
                );
        api.skipReportSummary(null);
    }

    private ReportDefinitionMap getReportDefinitionMap() {
        return new ReportDefinitionMap()
                .withDefaultCustomReportWithUniqueReportName()
                .withCustomDatesAndCampaignIds(DATE, DATE, CAMPAIGN_ID)
                .withFieldNames(FieldEnum.IMPRESSIONS);
    }

}