package ru.yandex.autotests.directapi.reports.backtoback;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.annotation.XmlAnyElement;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import com.google.gson.FieldNamingPolicy;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializer;
import com.google.gson.TypeAdapter;
import com.yandex.direct.api.v5.reports.DateRangeTypeEnum;
import com.yandex.direct.api.v5.reports.FieldEnum;
import com.yandex.direct.api.v5.reports.FilterOperatorEnum;
import com.yandex.direct.api.v5.reports.FormatEnum;
import com.yandex.direct.api.v5.reports.ReportDefinition;
import com.yandex.direct.api.v5.reports.ReportTypeEnum;
import org.apache.commons.lang.RandomStringUtils;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runners.Parameterized;

import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsData;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.json.HashCodeCalcExclusionStrategy;
import ru.yandex.autotests.irt.testutils.json.JAXBElementSerializer;
import ru.yandex.qatools.elliptics.ElClient;

import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by andy-ilyin on 01.12.16.
 */
public abstract class TestBase {
    private static LogSteps log = LogSteps.getLogger(TestBase.class);

    private static final String ELLIPTICS_NAMESPACE = "direct-reports5";
    private static final Pattern REQUESTS_FILE_NAME_PATTERN = Pattern.compile("^(.*)\\.txt$");
    private static final Pattern AQUA_FILE_GET_PREFIX_REGEX = Pattern.compile("^https?://aqua\\.yandex-team\\.ru/storage/get");

    private static final TypeAdapter<ReportDefinition> GSON_ADAPTER = new GsonBuilder()
            .serializeSpecialFloatingPointValues()
            .setFieldNamingPolicy(FieldNamingPolicy.UPPER_CAMEL_CASE)
            .setExclusionStrategies(new HashCodeCalcExclusionStrategy())
            .setDateFormat(new SimpleDateFormat("yyyy-MM-dd").toPattern())
            .registerTypeAdapter(JAXBElement.class, new JAXBElementSerializer())
            .registerTypeHierarchyAdapter(Enum.class,
                    (JsonDeserializer<Enum>) (json, typeOfT, context) -> {
                        if (json == null) {
                            return null;
                        }

                        // выбросит IllegalArgumentException, если в enum нет такого варианта
                        return Enum.valueOf((Class) typeOfT, json.getAsString());
                    })
            .setPrettyPrinting()
            .create()
            .getAdapter(ReportDefinition.class);

    // TODO: конфиг-опцию, чтобы можно было указывать нужные файлы в пропертях; сейчас выполняются
    // TODO: только файлы с runByDefault = true
    protected enum TestFile {
        FIELD_NAMES_BY_ONE("fieldnamesbyone.txt"),
        FILTERING_BY_EVERY_FIELD("filteringbyeveryfield.txt"),
        FILTERING_BY_EVERY_FIELD_WITH_EVERY_OPERATOR("filteringbyeveryfieldwitheveryoperator.txt"),
        FILTERING_BY_EVERY_FIELD_WITH_EVERY_OPERATOR_FAIL("filteringbyeveryfieldwitheveryoperatorFAIL.txt", false),
        INCLUDE_DISCOUNT_AND_VAT("includediscountandvat.txt"),
        ONE_REQUEST("onerequest.txt"),
        SOME_FAT_REQUESTS("somefatrequests.txt", false),
        SORTING_BY_EVERY_FIELD("sortingbyeveryfield.txt"),
        SORTING_BY_EVERY_FIELD_FAIL("sortingbyeveryfieldFAIL.txt", false),
        SORTING_BY_GOALS_AND_ATTRIBUTION_MODELS("goalsandattributionmodels.txt"),

        AD_PERFORMANCE_REPORT("report_types/AD_PERFORMANCE_REPORT.txt"),
        ACCOUNT_PERFORMANCE_REPORT("report_types/ACCOUNT_PERFORMANCE_REPORT.txt"),
        ADGROUP_PERFORMANCE_REPORT("report_types/ADGROUP_PERFORMANCE_REPORT.txt"),
        CAMPAIGN_PERFORMANCE_REPORT("report_types/CAMPAIGN_PERFORMANCE_REPORT.txt"),
        CRITERIA_PERFORMANCE_REPORT("report_types/CRITERIA_PERFORMANCE_REPORT.txt"),
        CRITERIA_PERFORMANCE_REPORT_FAIL("report_types/CRITERIA_PERFORMANCE_REPORT_FAIL.txt", false),
        CUSTOM_REPORT("report_types/CUSTOM_REPORT.txt"),
        REACH_AND_FREQUENCY_PERFORMANCE_REPORT("report_types/REACH_AND_FREQUENCY_PERFORMANCE_REPORT.txt");

        final String fileName;
        final String fileNameWithoutExtension;
        final boolean runByDefault;

        TestFile(String fileName) {
            this(fileName, true);
        }

        TestFile(String fileName, boolean runByDefault) {
            this.fileName = fileName;
            this.runByDefault = runByDefault;

            Matcher requestsFileNameMatcher = REQUESTS_FILE_NAME_PATTERN.matcher(fileName);
            if (!requestsFileNameMatcher.matches()) {
                throw new IllegalArgumentException("неправильный формат имени файла с запросами");
            }

            this.fileNameWithoutExtension = requestsFileNameMatcher.group(1);
        }
    }

    static class TestCase {
        final TestFile testFile;
        final long lineNumber;
        final String requestJson;
        final String clientLogin;

        TestCase(TestFile testFile, long lineNumber, String requestJson, String clientLogin) {
            this.testFile = testFile;
            this.lineNumber = lineNumber;
            this.requestJson = requestJson;
            this.clientLogin = clientLogin;
        }

        @Override
        public String toString() {
            if (clientLogin != null) {
                return String.format("%s, line %d login %s", testFile.fileName, lineNumber, clientLogin);
            } else {
                return String.format("%s, line %d", testFile.fileName, lineNumber);
            }
        }
    }

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER);

    @Parameterized.Parameter
    public TestCase testCase;

    private static Map<Long, String> campaignIdToLoginCache = new HashMap<>();

    protected String clientLogin;

    protected ReportDefinitionMap reportRequest;

    private static String ellipticsResultDirectoryBase;
    private static String ellipticsResponsesDirectoryBase;
    private static String ellipticsFailedTestsFileName;
    private static boolean onlyRunFailedTests;
    private static String previouslyFailedTestsLocation;
    private static String ellipticsTemporaryDirectory;

    static {
        DirectTestRunProperties properties = DirectTestRunProperties.getInstance();

        String todayDate = new SimpleDateFormat("yyyy-MM-dd").format(new Date());

        ellipticsTemporaryDirectory = String.format("%s/%s-%s",
                ELLIPTICS_NAMESPACE, todayDate, RandomStringUtils.randomAlphabetic(5));

        ellipticsResultDirectoryBase = properties.getApi5ReportsBacktobackResultDirectory();
        if (ellipticsResultDirectoryBase == null ||
                ellipticsResultDirectoryBase.isEmpty())
        {
            ellipticsResultDirectoryBase = todayDate + "-" + RandomStringUtils.randomAlphabetic(5);
        }

        if (properties.isApi5ReportsBacktobackResponsesInIndefiniteStorage()) {
            if (properties.getApi5ReportsBacktobackResponsesDirectory() != null &&
                    !properties.getApi5ReportsBacktobackResponsesDirectory().isEmpty())
            {
                throw new IllegalArgumentException("Api5ReportsBacktobackResponsesInIndefiniteStorage несовместимо с Api5ReportsBacktobackResponsesDirectory");
            }

            ellipticsResponsesDirectoryBase = ElClient.INDEFINITELY + ELLIPTICS_NAMESPACE;
        } else {
            ellipticsResponsesDirectoryBase = ELLIPTICS_NAMESPACE + "/" + ellipticsResultDirectoryBase;
        }

        onlyRunFailedTests = properties.isApi5ReportsBacktobackCompareOnlyRunFailed();

        ellipticsFailedTestsFileName = String.format("%s/%s/failed.txt",
                ELLIPTICS_NAMESPACE, ellipticsResultDirectoryBase);

        String failedUrlProperty = properties.getApi5ReportsBacktobackCompareFailedUrl();
        if (failedUrlProperty != null && !failedUrlProperty.isEmpty()) {
            String failedUrl = failedUrlProperty;

            failedUrl = AQUA_FILE_GET_PREFIX_REGEX.matcher(failedUrl).replaceFirst("");

            if (failedUrl.startsWith("/")) {
                failedUrl = failedUrl.substring(1);
            }

            previouslyFailedTestsLocation = failedUrl;
        } else {
            previouslyFailedTestsLocation = ellipticsFailedTestsFileName;
        }

    }

    static List<TestFile> getTestFilesToRunBasedOnProperties() {
        DirectTestRunProperties properties = DirectTestRunProperties.getInstance();

        String testFilesProperty = properties.getApi5ReportsBacktobackTestfiles();

        List<TestFile> testFilesToRun;

        if (testFilesProperty != null && !testFilesProperty.isEmpty()) {
            testFilesToRun = Arrays.stream(testFilesProperty.split("[, ]"))
                    .map(TestFile::valueOf)
                    .collect(Collectors.toList());
        } else {
            testFilesToRun = Arrays.stream(TestFile.values())
                    .filter(testFile -> testFile.runByDefault)
                    .collect(Collectors.toList());
        }
        return testFilesToRun;
    }

    protected static Collection<Object[]> calculateParameters(List<TestFile> testFilesToRun) {
        Collection<Object[]> result = new ArrayList<>();

        for (TestFile testFile : testFilesToRun) {
            File f = new File(TestBase.class.getClassLoader()
                    .getResource("backtoback/requests/" + testFile.fileName)
                    .getFile());

            long lineNumber = 0;
            String currentClientLogin = null;

            try (Scanner s = new Scanner(f)) {
                while (s.hasNextLine()) {
                    lineNumber++;

                    String line = s.nextLine();

                    // комментарий
                    if (line.startsWith("//")) {
                        continue;
                    }

                    // пустая строка
                    if (line.matches("^\\s*$")) {
                        continue;
                    }

                    // логин для последующих запросов
                    if (line.startsWith("#login ")) {
                        currentClientLogin = line.replaceFirst("#login ", "");
                        continue;
                    }

                    result.add(new Object[]{new TestCase(testFile, lineNumber, line, currentClientLogin)});
                }

                s.close();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        return result;
    }

    @Before
    public void prepareRequest() {
        ReportDefinition reportDefinition;
        try {
            reportDefinition = GSON_ADAPTER.fromJson(testCase.requestJson);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        reportRequest = new ReportDefinitionMap(reportDefinition)
                .withFormat(FormatEnum.TSV);

        if (reportDefinition.getReportType() == null) {
            reportDefinition.setReportType(ReportTypeEnum.CUSTOM_REPORT);
        }

        reportRequest.withUniqueReportName();

        clientLogin = testCase.clientLogin;
        if (clientLogin == null) {
            List<Long> campaignIds = new ArrayList<>();

            reportDefinition.getSelectionCriteria().getFilter().stream()
                    .filter(filterItem ->
                            filterItem.getField() == FieldEnum.CAMPAIGN_ID && (
                                    filterItem.getOperator() == FilterOperatorEnum.EQUALS ||
                                            filterItem.getOperator() == FilterOperatorEnum.IN))
                    .map(filterItem -> filterItem.getValues().stream().map(Long::parseLong).collect(Collectors.toList()))
                    .forEach(campaignIds::addAll);

            assumeThat("список кампаний в запросе не пуст", campaignIds, not(empty()));
            assumeThat("в запросе кастомный диапазон дат",
                    reportDefinition.getDateRangeType(), equalTo(DateRangeTypeEnum.CUSTOM_DATE));

            long campaignId = campaignIds.get(0);

            log.info("getting login for campaignId = " + campaignId);

            if (campaignIdToLoginCache.containsKey(campaignId)) {
                clientLogin = campaignIdToLoginCache.get(campaignId);
                log.info("got from cache: " + clientLogin);
            } else {
                DirectJooqDbSteps dbSteps = api.userSteps.getDirectJooqDbSteps();
                dbSteps.useShard(dbSteps.shardingSteps().getShardByCid(campaignId));
                CampaignsRecord campaignsRecord = dbSteps.campaignsSteps().getCampaignById(campaignId);

                clientLogin = dbSteps.usersSteps().getUsers(campaignsRecord.getUid()).getLogin();
                campaignIdToLoginCache.put(campaignId, clientLogin);
                log.info("got from db: " + clientLogin);
            }
        }
    }

    protected final String getEllipticsRequestsFileName() {
        return String.format("%s/%s/responses/%s.xml",
                ellipticsResponsesDirectoryBase,
                testCase.testFile.fileNameWithoutExtension, testCase.lineNumber);
    }

    protected final String getEllipticsDiffFileName() {
        return String.format("%s/%s/%s/differences/%s.diff.html",
                ELLIPTICS_NAMESPACE, ellipticsResultDirectoryBase,
                testCase.testFile.fileNameWithoutExtension, testCase.lineNumber);
    }

    protected final String getEllipticsActualResultsFileName() {
        return String.format("%s/%s/%s.xml",
                ellipticsTemporaryDirectory,
                testCase.testFile.fileNameWithoutExtension, testCase.lineNumber);
    }

    protected static String getEllipticsFailedTestsFileName() {
        return ellipticsFailedTestsFileName;
    }

    public static boolean onlyRunFailedTests() {
        return onlyRunFailedTests;
    }

    public static String getPreviouslyFailedTestsLocation() {
        return previouslyFailedTestsLocation;
    }

    @XmlRootElement(name="field")
    public static class ReportXmlRowField {
        @XmlAttribute
        public String name;

        @XmlAttribute
        public String value;

        public ReportXmlRowField() { }

        public ReportXmlRowField(String name, String value) {
            this.name = name;
            this.value = value;
        }
    }

    @XmlRootElement(name="row")
    public static class ReportXmlRow {
        @XmlAnyElement
        public List<ReportXmlRowField> values;

        public ReportXmlRow() { }

        public ReportXmlRow(List<ReportXmlRowField> values) {
            this.values = values;
        }
    }

    @XmlRootElement(name="report")
    public static class ReportXmlRepresentation {
        @XmlAttribute
        public String headerStartDate;

        @XmlAttribute
        public String headerEndDate;

        @XmlElement
        public List<String> columns;

        @XmlAnyElement
        public List<ReportXmlRow> rows;

        @XmlAttribute
        public long rowCount;

        public ReportXmlRepresentation() {}

        public ReportXmlRepresentation(String headerStartDate,
                                       String headerEndDate,
                                       List<String> columns,
                                       List<ReportXmlRow> rows,
                                       long rowCount)
        {
            this.headerStartDate = headerStartDate;
            this.headerEndDate = headerEndDate;
            this.columns = columns;
            this.rows = rows;
            this.rowCount = rowCount;
        }
    }

    protected static String getComparableReportString(ReportsData reportsData) {
        List<ReportXmlRow> xmlRows = reportsData.getReportsLines().stream()
                .map(row -> new ReportXmlRow(row.getRawFields().entrySet().stream()
                        .map(fieldEntry -> new ReportXmlRowField(fieldEntry.getKey().value(), fieldEntry.getValue()))
                        .collect(Collectors.toList())))
                .collect(Collectors.toList());

        List<String> columns = reportsData.getColumnsHeadersList();

        ReportXmlRepresentation xmlRepresentation = new ReportXmlRepresentation(
                reportsData.getReportHeaderStartDate(),
                reportsData.getReportHeaderEndDate(),
                columns,
                xmlRows,
                reportsData.getTotalRowsNumber());

        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(
                    ReportXmlRowField.class,
                    ReportXmlRow.class,
                    ReportXmlRepresentation.class);

            Marshaller jaxbMarshaller = jaxbContext.createMarshaller();
            jaxbMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
            StringWriter stringWriter = new StringWriter();
            jaxbMarshaller.marshal(xmlRepresentation, stringWriter);
            return stringWriter.toString();
        } catch (JAXBException e) {
            throw new RuntimeException(e);
        }
    }

    @Test
    public abstract void test();
}
