package ru.yandex.autotests.directapi.reports.criteriareport;

import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.general.SortOrderEnum;
import com.yandex.direct.api.v5.reports.DateRangeTypeEnum;
import com.yandex.direct.api.v5.reports.FieldEnum;
import com.yandex.direct.api.v5.reports.FilterOperatorEnum;
import com.yandex.direct.api.v5.reports.ReportTypeEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.model.api5.reports.FilterItemMap;
import ru.yandex.autotests.directapi.model.api5.reports.OrderByMap;
import ru.yandex.autotests.directapi.model.api5.reports.PageMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsData;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsLine;
import ru.yandex.autotests.directapi.model.api5.reports.SelectionCriteriaMap;
import ru.yandex.autotests.directapi.reports.ReportsFeatures;
import ru.yandex.autotests.directapi.reports.ReportsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.irt.testutils.matchers.OrderMatcher;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;

import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by andy-ilyin on 13.01.17.
 */
@Aqua.Test
@Description("CRITERIA_PERFORMANCE_REPORT: OrderBy, положительные сценарии")
@Issue("https://st.yandex-team.ru/TESTIRT-10893")
@Features(ReportsFeatures.CRITERIA_PERFORMANCE_REPORT)
@RunWith(Parameterized.class)
public class CriteriaPerformanceReportOrderByPositiveTest {
    private static final EnumSet<FieldEnum> DATE_FIELDS = EnumSet.of(
            FieldEnum.DATE,
            FieldEnum.WEEK,
            FieldEnum.MONTH,
            FieldEnum.QUARTER,
            FieldEnum.YEAR);

    private static final FilterItemMap FILTER = new FilterItemMap()
            .withField(FieldEnum.CAMPAIGN_ID)
            .withOperator(FilterOperatorEnum.IN)
            .withValues(35058347L, 35056372L);

    private static final FilterItemMap SYZRAN_FILTER = new FilterItemMap()
            .withField(FieldEnum.LOCATION_OF_PRESENCE_ID)
            .withOperator(FilterOperatorEnum.NOT_EQUALS)
            .withValues(11139L);

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER).clientLogin(ReportsLogins.CLIENT_FOR_CRITERIA_PERFORMANCE_REPORT4);

    @Parameterized.Parameter
    public FieldEnum fieldName;

    @Parameterized.Parameter(1)
    public Function<ReportsLine, ?> valueGetter;

    @Parameterized.Parameters(name="{0}")
    public static Collection<Object[]> parameters() {
        Object[][] result = new Object[][] {
                new Object[] { FieldEnum.AD_GROUP_ID, (Function<ReportsLine, ?>) ReportsLine::getAdGroupId },
                new Object[] { FieldEnum.AD_GROUP_NAME, (Function<ReportsLine, ?>) ReportsLine::getAdGroupName },
                new Object[] { FieldEnum.AD_NETWORK_TYPE, (Function<ReportsLine, ?>) ReportsLine::getAdNetworkType },
                new Object[] { FieldEnum.AGE, (Function<ReportsLine, ?>) ReportsLine::getAge },
                new Object[] { FieldEnum.AVG_CLICK_POSITION, (Function<ReportsLine, ?>) ReportsLine::getAvgClickPosition },
                new Object[] { FieldEnum.AVG_CPC, (Function<ReportsLine, ?>) ReportsLine::getAvgCpc },
                new Object[] { FieldEnum.AVG_IMPRESSION_POSITION, (Function<ReportsLine, ?>) ReportsLine::getAvgImpressionPosition },
                new Object[] { FieldEnum.AVG_PAGEVIEWS, (Function<ReportsLine, ?>) ReportsLine::getAvgPageviews },
                new Object[] { FieldEnum.BOUNCE_RATE, (Function<ReportsLine, ?>) ReportsLine::getBounceRate },
                new Object[] { FieldEnum.BOUNCES, (Function<ReportsLine, ?>) ReportsLine::getBounces },
                new Object[] { FieldEnum.CAMPAIGN_ID, (Function<ReportsLine, ?>) ReportsLine::getCampaignId },
                new Object[] { FieldEnum.CAMPAIGN_NAME, (Function<ReportsLine, ?>) ReportsLine::getCampaignName },
                new Object[] { FieldEnum.CARRIER_TYPE, (Function<ReportsLine, ?>) ReportsLine::getCarrierType },
                new Object[] { FieldEnum.CLICK_TYPE, (Function<ReportsLine, ?>) ReportsLine::getClickType },
                new Object[] { FieldEnum.CLICKS, (Function<ReportsLine, ?>) ReportsLine::getClicks },
                new Object[] { FieldEnum.COST, (Function<ReportsLine, ?>) ReportsLine::getCost },

                // TODO: здесь сортировка работает плохо, но менеджера по продукту это устраивает
//                new Object[] { FieldEnum.CRITERIA_ID, (Function<ReportsLine, ?>) ReportsLine::getCriteriaId },

                new Object[] { FieldEnum.CTR, (Function<ReportsLine, ?>) ReportsLine::getCtr },
                new Object[] { FieldEnum.DATE, (Function<ReportsLine, ?>) ReportsLine::getDate },
                new Object[] { FieldEnum.DEVICE, (Function<ReportsLine, ?>) ReportsLine::getDevice },
                new Object[] { FieldEnum.EXTERNAL_NETWORK_NAME, (Function<ReportsLine, ?>) ReportsLine::getExternalNetworkName },
                new Object[] { FieldEnum.GENDER, (Function<ReportsLine, ?>) ReportsLine::getGender },
                new Object[] { FieldEnum.IMPRESSIONS, (Function<ReportsLine, ?>) ReportsLine::getImpressions },
                new Object[] { FieldEnum.LOCATION_OF_PRESENCE_ID, (Function<ReportsLine, ?>) ReportsLine::getLocationOfPresenceId },
                new Object[] { FieldEnum.LOCATION_OF_PRESENCE_NAME, (Function<ReportsLine, ?>) ReportsLine::getLocationOfPresenceName },
                new Object[] { FieldEnum.MOBILE_PLATFORM, (Function<ReportsLine, ?>) ReportsLine::getMobilePlatform },
                new Object[] { FieldEnum.PROFIT, (Function<ReportsLine, ?>) ReportsLine::getProfit },
                new Object[] { FieldEnum.SESSIONS, (Function<ReportsLine, ?>) ReportsLine::getSessions },
                new Object[] { FieldEnum.SLOT, (Function<ReportsLine, ?>) ReportsLine::getSlot },
                new Object[] { FieldEnum.TARGETING_LOCATION_ID, (Function<ReportsLine, ?>) ReportsLine::getTargetingLocationId },
                new Object[] { FieldEnum.TARGETING_LOCATION_NAME, (Function<ReportsLine, ?>) ReportsLine::getTargetingLocationName },
                new Object[] { FieldEnum.WEEK, (Function<ReportsLine, ?>) ReportsLine::getWeek },
        };

        return Arrays.asList(result);
    }

    private FieldEnum[] requestedFieldNames;
    private FilterItemMap[] filters;

    @Before
    public void prepare() {
        if (DATE_FIELDS.contains(fieldName)) {
            requestedFieldNames = new FieldEnum[] { fieldName, FieldEnum.AD_GROUP_ID };
        } else {
            requestedFieldNames = new FieldEnum[] { FieldEnum.DATE, fieldName };
        }

        if (fieldName.equals(FieldEnum.LOCATION_OF_PRESENCE_NAME) || fieldName.equals(FieldEnum.TARGETING_LOCATION_NAME)) {
            filters = new FilterItemMap[] { FILTER, SYZRAN_FILTER };
        } else {
            filters = new FilterItemMap[] { FILTER };
        }
    }

    @Test
    public void test() {
        ReportsData report = api.userSteps.reportsSteps().callReportsXml(new ReportDefinitionMap()
                .withDefaultReportWithTypeAndUniqueReportName(ReportTypeEnum.CRITERIA_PERFORMANCE_REPORT)
                .withDateRangeType(DateRangeTypeEnum.CUSTOM_DATE)
                .withSelectionCriteria(new SelectionCriteriaMap()
                        .withDateFrom("2021-11-06")
                        .withDateTo("2021-11-13")
                        .withFilter(filters))
                .withFieldNames(requestedFieldNames)
                .withOrderBy(new OrderByMap().withField(fieldName))
                .withPage(new PageMap().withLimit(1500L)));

        assumeThat("отчёт не пустой", report.getReportsLines(), not(empty()));

        List<Object> columnData = report.getReportsLines().stream()
                .map(valueGetter)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        assumeThat("в отчёте по меньшей мере два разных значения, отличных от null",
                columnData.stream().distinct().collect(Collectors.toList()),
                hasSize(greaterThanOrEqualTo(2)));

        assertThat("значения упорядочены", columnData, OrderMatcher.isAscendingOrdered());
    }

    @Test
    public void testAscending() {
        ReportsData report = api.userSteps.reportsSteps().callReportsXml(new ReportDefinitionMap()
                .withDefaultReportWithTypeAndUniqueReportName(ReportTypeEnum.CRITERIA_PERFORMANCE_REPORT)
                .withDateRangeType(DateRangeTypeEnum.CUSTOM_DATE)
                .withSelectionCriteria(new SelectionCriteriaMap()
                        .withDateFrom("2021-11-06")
                        .withDateTo("2021-11-13")
                        .withFilter(filters))
                .withFieldNames(requestedFieldNames)
                .withOrderBy(new OrderByMap()
                        .withField(fieldName)
                        .withSortOrder(SortOrderEnum.ASCENDING))
                .withPage(new PageMap().withLimit(1500L)));

        assumeThat("отчёт не пустой", report.getReportsLines(), not(empty()));

        List<Object> columnData = report.getReportsLines().stream()
                .map(valueGetter)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        assumeThat("в отчёте по меньшей мере два разных значения, отличных от null",
                columnData.stream().distinct().collect(Collectors.toList()),
                hasSize(greaterThanOrEqualTo(2)));

        assertThat("значения упорядочены", columnData, OrderMatcher.isAscendingOrdered());
    }

    @Test
    public void testDescending() {
        ReportsData report = api.userSteps.reportsSteps().callReportsXml(new ReportDefinitionMap()
                .withDefaultReportWithTypeAndUniqueReportName(ReportTypeEnum.CRITERIA_PERFORMANCE_REPORT)
                .withDateRangeType(DateRangeTypeEnum.CUSTOM_DATE)
                .withSelectionCriteria(new SelectionCriteriaMap()
                        .withDateFrom("2021-11-06")
                        .withDateTo("2021-11-13")
                        .withFilter(filters))
                .withFieldNames(requestedFieldNames)
                .withOrderBy(new OrderByMap()
                        .withField(fieldName)
                        .withSortOrder(SortOrderEnum.DESCENDING))
                .withPage(new PageMap().withLimit(1500L)));

        assumeThat("отчёт не пустой", report.getReportsLines(), not(empty()));

        List<Object> columnData = report.getReportsLines().stream()
                .map(valueGetter)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        assumeThat("в отчёте по меньшей мере два разных значения, отличных от null",
                columnData.stream().distinct().collect(Collectors.toList()),
                hasSize(greaterThanOrEqualTo(2)));

        assertThat("значения упорядочены", columnData, OrderMatcher.isDescendingOrdered());
    }
}
