package ru.yandex.autotests.directapi.reports.daterange;

import org.apache.commons.lang.WordUtils;
import org.joda.time.DateTime;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsData;
import ru.yandex.autotests.directapi.model.api5.reports.SelectionCriteriaMap;
import ru.yandex.autotests.directapi.reports.ReportsFeatures;
import ru.yandex.autotests.directapi.reports.ReportsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import java.util.Arrays;
import java.util.Collection;

/**
 * Created by gerdler on 20.10.16.
 * https://st.yandex-team.ru/TESTIRT-10387
 */
@Aqua.Test
@Description("Негативные тесты валидации дат в отчетах с произвольными датами (CUSTOM_DATE)")
@Issue("https://st.yandex-team.ru/DIRECT-56509")
@Features(ReportsFeatures.ONLINE_CUSTOM_REPORT)
@RunWith(Parameterized.class)
public class DateValidationCustomRangeNegativeTest {

    private static final Api5Error ERROR_INCORRECT_FORMAT_DATE_FROM =
            new Api5Error(4001, Api5ErrorDetails.INCORRECT_DATE_FORMAT, WordUtils.capitalize(SelectionCriteriaMap.DATE_FROM));
    private static final Api5Error ERROR_INCORRECT_FORMAT_DATE_TO =
            new Api5Error(4001, Api5ErrorDetails.INCORRECT_DATE_FORMAT, WordUtils.capitalize(SelectionCriteriaMap.DATE_TO));
    private static final Api5Error ERROR_DATE_TO_EARLY_DATE_FROM =
            new Api5Error(4001, Api5ErrorDetails.DATE_TO_CANT_BE_EARLY_DATE_FROM);
    private static final Api5Error ERROR_DATE_FROM_IN_FUTURE =
            new Api5Error(4001, Api5ErrorDetails.DATE_IN_FUTURE, WordUtils.capitalize(SelectionCriteriaMap.DATE_FROM));

    private static final String VALID_DATE = "2016-10-01";
    private static final String LATER_VALID_DATE = "2016-10-02";
    private static final String DATE_IN_FUTURE = DateTime.now().plusDays(1).toString(ReportsData.DATE_FORMAT);
    private static final String DATE_WITHOUT_HYPHENS = "20161001";
    private static final String DATE_WITH_INVALID_MONTH = "2016-14-01";
    private static final String DATE_WITH_INVALID_DAY = "2016-10-38";
    private static final String DATE_ONLY_LETTERS = "today";
    private static final String DATE_ONLY_ZEROS = "0000-00-00";
    private static final String EMPTY_DATE = "";
    private static final String DATE_ONLY_SPACE = " ";

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER).clientLogin(ReportsLogins.REPORT_CLIENT_FOR_INVALID_DATE_FORMAT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String dateFrom;

    @Parameterized.Parameter(2)
    public String dateTo;

    @Parameterized.Parameter(3)
    public Api5Error expectedError;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                { "дата без дефисов в поле DateFrom",
                        DATE_WITHOUT_HYPHENS, VALID_DATE, ERROR_INCORRECT_FORMAT_DATE_FROM },
                { "дата без дефисов в поле DateTo",
                        VALID_DATE, DATE_WITHOUT_HYPHENS, ERROR_INCORRECT_FORMAT_DATE_TO },

                { "дата без дефисов в полях DateFrom и DateTo одновременно",
                        DATE_WITHOUT_HYPHENS, DATE_WITHOUT_HYPHENS, ERROR_INCORRECT_FORMAT_DATE_FROM },

                { "дата в поле DateFrom позже даты в DateTo",
                        LATER_VALID_DATE, VALID_DATE, ERROR_DATE_TO_EARLY_DATE_FROM },

                { "дата в поле DateFrom в будущем",
                        DATE_IN_FUTURE, DATE_IN_FUTURE, ERROR_DATE_FROM_IN_FUTURE },

                { "дата с неправильным номером месяца в поле DateFrom",
                        DATE_WITH_INVALID_MONTH, VALID_DATE, ERROR_INCORRECT_FORMAT_DATE_FROM },
                { "дата с неправильным номером месяца в поле DateTo",
                        VALID_DATE, DATE_WITH_INVALID_MONTH, ERROR_INCORRECT_FORMAT_DATE_TO },

                { "дата с неправильным номером дня в месяце в поле DateFrom",
                        DATE_WITH_INVALID_DAY, VALID_DATE, ERROR_INCORRECT_FORMAT_DATE_FROM },
                { "дата с неправильным номером дня в месяце в поле DateTo",
                        VALID_DATE, DATE_WITH_INVALID_DAY, ERROR_INCORRECT_FORMAT_DATE_TO },

                { "дата с буквами в поле DateFrom",
                        DATE_ONLY_LETTERS, VALID_DATE, ERROR_INCORRECT_FORMAT_DATE_FROM },
                { "дата с буквами в поле DateTo",
                        VALID_DATE, DATE_ONLY_LETTERS, ERROR_INCORRECT_FORMAT_DATE_TO },

                { "дата с нулевыми значениями года месяца и дня в поле DateFrom",
                        DATE_ONLY_ZEROS, VALID_DATE, ERROR_INCORRECT_FORMAT_DATE_FROM },
                { "дата с нулевыми значениями года месяца и дня в поле DateTo",
                        VALID_DATE, DATE_ONLY_ZEROS, ERROR_INCORRECT_FORMAT_DATE_TO },

                { "пустая дата в поле DateFrom",
                        EMPTY_DATE, VALID_DATE, ERROR_INCORRECT_FORMAT_DATE_FROM },
                { "пустая дата в поле DateTo",
                        VALID_DATE, EMPTY_DATE, ERROR_INCORRECT_FORMAT_DATE_TO },

                { "дата состоящая из одного пробела в поле DateFrom",
                        DATE_ONLY_SPACE, VALID_DATE, ERROR_INCORRECT_FORMAT_DATE_FROM },
                { "дата состоящая из одного пробела в поле DateTo",
                        VALID_DATE, DATE_ONLY_SPACE, ERROR_INCORRECT_FORMAT_DATE_TO },
        });
    }

    @Test
    public void checkGettingErrorForCustomDate() {
        api.userSteps.reportsSteps().expectXmlErrorOnReports(
                new ReportDefinitionMap()
                        .withDefaultCustomReportWithUniqueReportName()
                        .withCustomDates(dateFrom, dateTo),
                expectedError
        );
    }
}
