package ru.yandex.autotests.directapi.reports.daterange;

import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.reports.DateRangeTypeEnum;
import com.yandex.direct.api.v5.reports.FieldEnum;
import com.yandex.direct.api.v5.reports.FormatEnum;
import com.yandex.direct.api.v5.reports.ReportTypeEnum;
import org.joda.time.DateTime;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsData;
import ru.yandex.autotests.directapi.model.api5.reports.SelectionCriteriaMap;
import ru.yandex.autotests.directapi.reports.ReportsFeatures;
import ru.yandex.autotests.directapi.reports.ReportsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import java.util.Arrays;
import java.util.Collection;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by onotole on 06.07.16.
 * https://st.yandex-team.ru/TESTIRT-9689
 */
@Aqua.Test
@Description("Проверка поведения для различных значений DateRangeType")
@Issue("https://st.yandex-team.ru/TESTIRT-9244")
@Features(ReportsFeatures.ONLINE_CUSTOM_REPORT)
@RunWith(Parameterized.class)
public class DateValidationFixedRangeTest {
    private static final String CLIENT = ReportsLogins.CLIENT_FOR_DATE_VALIDATION;

    private static final DateTime todayDate = DateTime.now();
    private static final DateTime lastFriday = todayDate.minusWeeks(todayDate.getDayOfWeek() > 5 ? 0 : 1).withDayOfWeek(5);
    private static final DateTime mondayBeforeLastFriday = lastFriday.withDayOfWeek(1);
    private static final DateTime sundayPreviousWeek = todayDate.minusWeeks(1).withDayOfWeek(7);
    private static final DateTime sunday2WeeksAgo = todayDate.minusWeeks(2).withDayOfWeek(7);

    private static final DateTime mondayPreviousWeek = todayDate.minusWeeks(1).withDayOfWeek(1);
    private static final DateTime mondayThisWeek = todayDate.withDayOfWeek(1);
    private static final DateTime saturdayPreviousWeek = todayDate.minusWeeks(1).withDayOfWeek(6);
    private static final DateTime firstDayOfThisMonth = todayDate.withDayOfMonth(1);
    private static final DateTime lastDayOfPreviousMonth = todayDate.withDayOfMonth(1).minusDays(1);
    private static final DateTime firstDayOfPreviousMonth = lastDayOfPreviousMonth.withDayOfMonth(1);
    private static final DateTime firstDayOfCollectingData = todayDate.withDayOfMonth(1).minusYears(3);

    private static final String lastDayOfPreviousMonthStr = lastDayOfPreviousMonth.toString(ReportsData.DATE_FORMAT);
    private static final String firstDayOfPreviousMonthStr = firstDayOfPreviousMonth.toString(ReportsData.DATE_FORMAT);
    private static final String firstDayOfThisMonthStr = firstDayOfThisMonth.toString(ReportsData.DATE_FORMAT);
    private static final String saturdayPreviousWeekStr = saturdayPreviousWeek.toString(ReportsData.DATE_FORMAT);
    private static final String sundayPreviousWeekStr = sundayPreviousWeek.toString(ReportsData.DATE_FORMAT);
    private static final String sunday2WeeksAgoStr = sunday2WeeksAgo.toString(ReportsData.DATE_FORMAT);
    private static final String mondayPreviousWeekStr = mondayPreviousWeek.toString(ReportsData.DATE_FORMAT);
    private static final String mondayThisWeekStr = mondayThisWeek.toString(ReportsData.DATE_FORMAT);
    private static final String lastFridayStr = lastFriday.toString(ReportsData.DATE_FORMAT);
    private static final String mondayBeforeLastFridayStr = mondayBeforeLastFriday.toString(ReportsData.DATE_FORMAT);
    private static final String todayStr = todayDate.toString(ReportsData.DATE_FORMAT);
    private static final String firstDayOfCollectingDataStr = firstDayOfCollectingData.toString(ReportsData.DATE_FORMAT);

    @Parameterized.Parameter(0)
    public DateRangeTypeEnum dateRangeType;

    @Parameterized.Parameter(1)
    public String expectedDateStart;

    @Parameterized.Parameter(2)
    public String expectedDateFinish;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> testData() {
        return Arrays.asList(new Object[][] {
                {DateRangeTypeEnum.TODAY, todayStr, todayStr},
                {DateRangeTypeEnum.LAST_BUSINESS_WEEK, mondayBeforeLastFridayStr, lastFridayStr},
                {DateRangeTypeEnum.LAST_WEEK, mondayPreviousWeekStr, sundayPreviousWeekStr},
                {DateRangeTypeEnum.LAST_WEEK_SUN_SAT, sunday2WeeksAgoStr, saturdayPreviousWeekStr},
                {DateRangeTypeEnum.LAST_MONTH, firstDayOfPreviousMonthStr, lastDayOfPreviousMonthStr},
                {DateRangeTypeEnum.THIS_MONTH, firstDayOfThisMonthStr, todayStr},
                {DateRangeTypeEnum.THIS_WEEK_MON_TODAY, mondayThisWeekStr, todayStr},
                {DateRangeTypeEnum.THIS_WEEK_SUN_TODAY, sundayPreviousWeekStr, todayStr},
                {DateRangeTypeEnum.ALL_TIME, firstDayOfCollectingDataStr, todayStr},
        });
    }

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER).clientLogin(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Test
    public void checkDate() {
        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withFieldNames(FieldEnum.CAMPAIGN_ID)
                .withUniqueReportName()
                .withReportType(ReportTypeEnum.CUSTOM_REPORT)
                .withFormat(FormatEnum.TSV)
                .withIncludeVAT(YesNoEnum.YES)
                .withIncludeDiscount(YesNoEnum.YES)
                .withSelectionCriteria(new SelectionCriteriaMap())
                .withDateRangeType(dateRangeType);

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);
        String actualReportName = reportsData.getReportHeader();

        String expectedReportName = String.format(ReportsData.REPORT_NAME_FORMAT, reportDefinitionMap.getReportName(), expectedDateStart, expectedDateFinish);
        assertThat("даты отчета за указанную дату вернулись верные", actualReportName, equalTo(expectedReportName));
    }
}
