package ru.yandex.autotests.directapi.reports.fieldnames;

import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.reports.DateRangeTypeEnum;
import com.yandex.direct.api.v5.reports.FieldEnum;
import com.yandex.direct.api.v5.reports.FormatEnum;
import com.yandex.direct.api.v5.reports.ReportTypeEnum;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.SelectionCriteriaMap;
import ru.yandex.autotests.directapi.reports.ReportsFeatures;
import ru.yandex.autotests.directapi.reports.ReportsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static org.apache.commons.lang3.StringUtils.capitalize;

/**
 * Created by pavryabov on 01.07.16.
 * https://st.yandex-team.ru/TESTIRT-9654
 */
@Aqua.Test
@Description("Проверка валидации комбинации полей в FieldNames")
@Features(ReportsFeatures.ONLINE_CUSTOM_REPORT)
@RunWith(Parameterized.class)
public class FieldNamesFieldsValidationNegativeTest {

    private static final String CLIENT = ReportsLogins.CLIENT_FOR_FIELD_NAMES_VALIDATION_NEGATIVE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(ReportsLogins.SUPER_LOGIN).clientLogin(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter
    public List<FieldEnum> fieldNames;

    @Parameterized.Parameter(1)
    public Api5Error expectedError;

    public static final class TestCase {
        private final List<FieldEnum> fieldNames;
        private final int expectedErrorCode;
        private final Api5ErrorDetails expectedErrorDetails;
        private final List<String> expectedErrorDetailsParams;

        private TestCase(List<FieldEnum> fieldNames,
                         int expectedErrorCode,
                         Api5ErrorDetails expectedErrorDetails,
                         Object... expectedErrorDetailsParams) {
            this.fieldNames = Collections.unmodifiableList(new ArrayList<>(fieldNames));
            this.expectedErrorCode = expectedErrorCode;
            this.expectedErrorDetails = expectedErrorDetails;
            this.expectedErrorDetailsParams = Collections.unmodifiableList(
                    Stream.of(expectedErrorDetailsParams)
                            .map(Object::toString)
                            .collect(Collectors.toList()));
        }

        private final Api5Error getExpectedError() {
            return new Api5Error(
                    expectedErrorCode,
                    expectedErrorDetails,
                    expectedErrorDetailsParams.toArray(new Object[0]));
        }
    }

    private static List<TestCase> testCases;
    static {
        testCases = new LinkedList<>();

        // из полей даты может быть только одно
        List<FieldEnum> dateFields = Arrays.asList(
                FieldEnum.DATE,
                FieldEnum.WEEK,
                FieldEnum.MONTH,
                FieldEnum.QUARTER,
                FieldEnum.YEAR);

        for (FieldEnum dateField1: dateFields) {
            for (FieldEnum dateField2: dateFields) {
                if (dateField1.compareTo(dateField2) >= 0) continue;

                testCases.add(new TestCase(
                        Arrays.asList(dateField1, dateField2),
                        4000,
                        Api5ErrorDetails.ONLY_ONE_OF_THE_WEEK_DATE_QUARTER_YEAR_MONTH));
            }
        }

        // ClickType несовместимо с несколькими другими полями
        testCases.addAll(
                Stream.of(
                        FieldEnum.IMPRESSIONS,
                        FieldEnum.AVG_IMPRESSION_POSITION,
                        FieldEnum.CTR
                )
                        .map(field -> new TestCase(
                                Arrays.asList(FieldEnum.CLICK_TYPE, field),
                                4000,
                                Api5ErrorDetails.REPORTS_CLICK_TYPE_INCOMPATIBLE))
                        .collect(Collectors.toList()));

        List<FieldEnum> criterionFields = Arrays.asList(FieldEnum.CRITERION, FieldEnum.CRITERION_ID, FieldEnum.CRITERION_TYPE);
        String criterionFieldList = criterionFields.stream().map(FieldEnum::value).collect(Collectors.joining(", "));
        List<FieldEnum> criteriaFields = Arrays.asList(FieldEnum.CRITERIA, FieldEnum.CRITERIA_ID);
        for (FieldEnum criterionField: criterionFields) {
            for (FieldEnum criteriaField : criteriaFields) {
                testCases.add(
                        new TestCase(
                                Arrays.asList(criteriaField, criterionField),
                                4000,
                                Api5ErrorDetails.INCOMPATIBLE_FIELDS,
                                capitalize(ReportDefinitionMap.FIELD_NAMES),
                                criteriaField.value(),
                                criterionFieldList,
                                ReportTypeEnum.CUSTOM_REPORT
                        )
                );
            }
        }

        testCases.add(
                new TestCase(
                        Arrays.asList(FieldEnum.CRITERIA_TYPE, FieldEnum.CRITERION_TYPE),
                        4000,
                        Api5ErrorDetails.INCOMPATIBLE_FIELDS,
                        capitalize(ReportDefinitionMap.FIELD_NAMES),
                        FieldEnum.CRITERIA_TYPE.value(),
                        FieldEnum.CRITERION_TYPE.value(),
                        ReportTypeEnum.CUSTOM_REPORT
                )
        );

        // TODO: Условие подбора (корректировки) несовместимо с несколькими другими полями: DIRECT-58959

        // Placement требует среза по AdNetworkType
        // TODO: DIRECT-58961
//        testCases.add(
//                new TestCase(Arrays.asList(FieldEnum.PLACEMENT),
//                        4000,
//                        // TODO: правильное сообщение
//                        Api5ErrorDetails.EMPTY_STRING));

        testCases = Collections.unmodifiableList(testCases);
    }

    @Parameterized.Parameters(name = "FieldNames = {0}")
    public static Collection<Object[]> data() {
        return testCases.stream()
                .map(testCase -> new Object[]{ testCase.fieldNames, testCase.getExpectedError() } )
                .collect(Collectors.toList());
    }

    @Test
    public void expectErrorWithFieldNames() {
        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withFieldNames(fieldNames.toArray(new FieldEnum[0]))
                .withDateRangeType(DateRangeTypeEnum.LAST_MONTH)
                .withUniqueReportName()
                .withReportType(ReportTypeEnum.CUSTOM_REPORT)
                .withFormat(FormatEnum.TSV)
                .withIncludeVAT(YesNoEnum.YES)
                .withIncludeDiscount(YesNoEnum.YES)
                .withSelectionCriteria(new SelectionCriteriaMap());
        api.userSteps.reportsSteps().expectXmlErrorOnReports(reportDefinitionMap, expectedError);
    }
}
